<?php

namespace QUI\QueueManager;

use QUI;
use QUI\QueueManager\Interfaces\QueueServerInterface;
use QUI\Exception;
use QUI\QueueManager\Interfaces\QueueServerProviderInterface;

use function class_exists;
use function is_a;

/**
 * Manages installed queue server implementations.
 */
class QueueManager
{
    /**
     * Get instance of queue server that is currently specified as the default server for this system.
     *
     * @return QueueServerInterface
     * @throws Exception
     */
    public function getCurrentQueueServer(): QueueServerInterface
    {
        return $this->getCurrentQueueServerProvider()->getQueueServer();
    }

    /**
     * Get instance of queue server provider that is currently specified as the default server for this system.
     *
     * @return QueueServerProviderInterface
     * @throws Exception
     */
    public function getCurrentQueueServerProvider(): QueueServerProviderInterface
    {
        try {
            $Conf = QUI::getPackage('quiqqer/queuemanager')->getConfig();
            $providerClass = $Conf->get('queue', 'server');
        } catch (\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
            throw new Exception('Cannot load queue server :: An unexpected error occurred.');
        }

        if (empty($providerClass)) {
            throw new Exception('Cannot load queue server :: No default server implementation specified.');
        }

        if (!class_exists($providerClass)) {
            throw new Exception(
                'Cannot load queue server :: Default server class ' . $providerClass . ' does not exist.'
            );
        }

        if (!is_a($providerClass, QueueServerProviderInterface::class, true)) {
            throw new Exception(
                'Cannot load queue server :: Default server class ' . $providerClass . ' does not'
                . ' implement ' . QueueServerProviderInterface::class . ' interface.'
            );
        }

        return new $providerClass();
    }

    /**
     * @return QueueServerProviderInterface[]
     */
    public function getInstalledQueueServerProviders(): array
    {
        // @todo cache einbauen

        $packages = QUI::getPackageManager()->getInstalled();
        $instances = [];

        foreach ($packages as $installedPackage) {
            try {
                $Package = QUI::getPackage($installedPackage['name']);

                if (!$Package->isQuiqqerPackage()) {
                    continue;
                }

                $packageProvider = $Package->getProvider();

                if (empty($packageProvider['queueServer'])) {
                    continue;
                }

                foreach ($packageProvider['queueServer'] as $class) {
                    if (!\class_exists($class)) {
                        continue;
                    }

                    if (!\is_a($class, QueueServerProviderInterface::class, true)) {
                        continue;
                    }

                    $instances[] = new $class();
                }
            } catch (QUI\Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }

        return $instances;
    }
}
