<?php

namespace QUI\QueueManager;

use DateTime;
use Exception;
use QUI\QueueManager\Interfaces\QueueServerInterface;
use QUI\QueueManager\Interfaces\QuiqqerQueueWorkerInterface;

/**
 * A job worker executes a job based on its job data
 */
abstract class AbstractQuiqqerQueueWorker implements QuiqqerQueueWorkerInterface
{
    private readonly QueueServerInterface $server;
    protected ?QuiqqerWorkerQueueJob $job = null;

    /**
     * @throws \QUI\Exception
     */
    public function __construct()
    {
        $queueManager = new QueueManager();
        $this->server = $queueManager->getCurrentQueueServer();
    }

    /**
     * Execute job
     *
     * @param QuiqqerWorkerQueueJob $job
     * @return JobData|null
     */
    public function execute(QuiqqerWorkerQueueJob $job): ?JobData
    {
        $this->job = $job;
        return $job->data;
    }

    /**
     * Write log entry for the job that is currenty processed
     *
     * @param string $msg
     * @return void
     */
    public function writeLogEntry(string $msg): void
    {
        $this->job?->log->addEntry([
            'time' => date('Y.m.d H:i:s'),
            'msg' => $msg
        ]);
    }

    /**
     * Clone the job this worker is executing
     * and queue it in the server queue
     *
     * @param integer|null $priority (optional) - new job priority
     * @param DateTime|null $EarliestQueueDate (optional) - Date at which the job shall be queued/executed the earliest
     * @return QueueJobUUID - ID of cloned job
     * @throws Exception
     */
    public function cloneJob(int $priority = null, ?DateTime $EarliestQueueDate = null): QueueJobUUID
    {
        if (is_null($this->job)) {
            throw new \Exception('No job set.');
        }

        return $this->server->cloneJob($this->job->uuid, $priority, $EarliestQueueDate);
    }
}
