<?php

namespace QUI\QueueManager;

use QUI;
use QUI\QueueManager\Repository\QueueJobRepository;
use QUI\System\Console\Tools\MigrationV2;

use function json_decode;
use function json_encode;

class EventHandling
{
    public static function onQuiqqerMigrationV2(MigrationV2 $Console): void
    {
        $Console->writeLn('- Migrate QueueManager');

        $table = QUI::getDBTableName(QueueJobRepository::TBL_JOBS);
        $connection = QUI::getDataBaseConnection();
        $queryBuilder = QUI::getQueryBuilder();

        // Migration: INT/Unix-Timestamp -> DATETIME
        $fields = ['createTime', 'lastUpdateTime', 'earliestQueueDate'];

        foreach ($fields as $field) {
            // 1. Spalte zuerst auf VARCHAR(32) umstellen (nimmt alles an)
            $connection->executeStatement(
                "ALTER TABLE `$table` MODIFY `$field` VARCHAR(32) NULL DEFAULT NULL"
            );

            // 2. Gültige Unix-Timestamps in DATETIME-Strings umwandeln
            $connection->executeStatement(
                "UPDATE `$table` SET `$field` = FROM_UNIXTIME(`$field`) WHERE `$field` REGEXP '^[0-9]{10}$' AND CAST(`$field` AS UNSIGNED) > 1000000000 AND CAST(`$field` AS UNSIGNED) <= 2147483647"
            );

            // 3. Ungültige Werte auf NULL setzen (leere Strings, zu große/kleine Werte, nur Ziffern aber nicht plausibel)
            $connection->executeStatement(
                "UPDATE `$table` SET `$field` = NULL WHERE `$field` = '' OR `$field` IS NULL OR `$field` = '0' OR `$field` < '1970-01-01 00:00:01' OR `$field` > '2100-01-01 00:00:00' OR `$field` REGEXP '^[0-9]+$'"
            );

            // 3b. Für NOT NULL Felder: NULLs durch Default ersetzen
            if (in_array($field, ['createTime', 'lastUpdateTime'])) {
                $connection->executeStatement(
                    "UPDATE `$table` SET `$field` = '1970-01-01 00:00:01' WHERE `$field` IS NULL"
                );
            }

            // 4. Spalte auf richtigen Typ umstellen
            if ($field === 'earliestQueueDate') {
                $connection->executeStatement(
                    "ALTER TABLE `$table` MODIFY `$field` DATETIME NULL DEFAULT NULL"
                );
            } else {
                $connection->executeStatement(
                    "ALTER TABLE `$table` MODIFY `$field` DATETIME NOT NULL"
                );
            }
        }

        // Change config if quiqqer/queueserver was set as current queue server provider
        $conf = QUI::getPackage('quiqqer/queuemanager')->getConfig();
        $confServer = $conf->get('queue', 'server');

        if (
            $confServer === 'QUI\QueueServer\Server' ||
            $confServer === '\QUI\QueueServer\Server'
        ) {
            $conf->set('queue', 'server', 'QUI\QueueServer\Provider');
            $conf->save();
        }

        $db = QUI::getDatabase();
        $tableHandler = $db->table();

        $select = [
            'id',
            'jobData',
            'uuid'
        ];

        $jobWorkerColumnExists = $tableHandler->existColumnInTable($table, 'jobWorker');

        if ($jobWorkerColumnExists) {
            $select[] = 'jobWorker';
        }

        $jobs = $db->fetchSQL("SELECT `" . implode('`,`', $select) . "` FROM $table");

        foreach ($jobs as $job) {
            if (empty($job['uuid'])) {
                $db->update(
                    $table,
                    [
                        'uuid' => QUI\Utils\Uuid::get()
                    ],
                    [
                        'id' => $job['id']
                    ]
                );
            }

            if ($jobWorkerColumnExists === false) {
                continue;
            }

            if (empty($job['jobWorker'])) {
                continue;
            }

            if (!empty($job['jobData'])) {
                $data = json_decode($job['jobData'], true);
            } else {
                $data = [];
            }

            $data['workerClass'] = $job['jobWorker'];

            $db->update(
                $table,
                [
                    'jobData' => json_encode($data),
                    'isQuiqqerWorkerJob' => 1
                ],
                [
                    'id' => $job['id']
                ]
            );
        }

        $tableHandler->deleteColumn($table, 'jobWorker');
    }
}
