<?php

namespace QUI\QueueManager\Interfaces;

use DateTime;
use QUI\QueueManager\JobData;
use QUI\QueueManager\JobLog;
use QUI\QueueManager\JobStatus;
use QUI\QueueManager\QueueJob;
use QUI\QueueManager\QueueJobUUID;

interface QueueServerInterface
{
    const JOB_STATUS_QUEUED = 1;
    const JOB_STATUS_RUNNING = 2;
    const JOB_STATUS_FINISHED = 3;
    const JOB_STATUS_ERROR = 4;
    const JOB_STATUS_CLONED = 5;

    /**
     * Adds a single job to the queue of a server
     *
     * @param QueueJob $job - The job to add to the queue
     * @return QueueJobUUID - unique Job ID
     */
    public function queueJob(QueueJob $job): QueueJobUUID;

    /**
     * Clone a job and queue it immediately
     *
     * @param QueueJobUUID $jobId - Job UUID
     * @param integer $priority - (new) job priority
     * @param DateTime|null $earliestQueueDate (optional) - Date at which the job shall be queued/executed the earliest
     * @return QueueJobUUID - ID of cloned job
     */
    public function cloneJob(QueueJobUUID $jobId, int $priority, ?DateTime $earliestQueueDate = null): QueueJobUUID;

    /**
     * Get status of a job
     *
     * @param QueueJobUUID $jobId
     * @return JobStatus
     */
    public function getJobStatus(QueueJobUUID $jobId): JobStatus;

    /**
     * Set status of a job
     *
     * @param QueueJobUUID $jobId
     * @param JobStatus $status
     * @return void
     */
    public function setJobStatus(QueueJobUUID $jobId, JobStatus $status): void;

    /**
     * Write log entry for a job
     *
     * @param QueueJobUUID $jobId
     * @param string $msg
     * @return void
     */
    public function writeJobLogEntry(QueueJobUUID $jobId, string $msg): void;

    /**
     * Get event log for specific job
     *
     * @param QueueJobUUID $jobId
     * @return JobLog
     */
    public function getJobLog(QueueJobUUID $jobId): JobLog;

    /**
     * Get result of a specific job
     *
     * @param QueueJobUUID $jobId
     * @param bool $deleteJob (optional) - delete job from queue after return [default: true]
     * @return JobData
     */
    public function getJobResult(QueueJobUUID $jobId, bool $deleteJob = true): JobData;

    /**
     * Set result of a specific job
     *
     * @param QueueJobUUID $jobId
     * @param JobData $result
     * @return void
     */
    public function setJobResult(QueueJobUUID $jobId, JobData $result): void;

    /**
     * Close server connection
     */
    public function closeConnection(): void;

    /**
     * Delete a job
     *
     * @param QueueJobUUID $jobId
     * @return void
     */
    public function deleteJob(QueueJobUUID $jobId): void;

//    /**
//     * Set status of a job
//     *
//     * @param integer $jobId
//     * @param integer $status
//     * @return bool - success
//     */
//    public function setJobStatus($jobId, $status);
}
