<?php

namespace QUI\QueueManager\Repository;

use DateTime;
use QUI;
use QUI\Database\DB;
use QUI\Database\Exception;
use QUI\QueueManager\Interfaces\QueueJobRepositoryInterface;
use QUI\QueueManager\JobData;
use QUI\QueueManager\JobStatus;
use QUI\QueueManager\QueueJob;
use QUI\QueueManager\QueueJobAttributes;
use QUI\QueueManager\QueueJobUUID;
use QUI\QueueManager\QuiqqerWorkerQueueJob;
use QUI\Utils\Grid;

use function current;
use function date_create;
use function is_null;
use function json_decode;
use function json_encode;

class QueueJobRepository implements QueueJobRepositoryInterface
{
    const TBL_JOBS = 'queueserver_jobs';

    public function __construct(private ?DB $db = null)
    {
        if (is_null($this->db)) {
            $this->db = QUI::getDataBase();
        }
    }

    /**
     * @param QueueJob $job
     * @return QueueJob - Inserted QueueJob with UUID
     *
     * @throws QUI\Database\Exception
     * @throws QUI\Exception
     */
    public function insert(QueueJob $job): QueueJob
    {
        if ($job->uuid) {
            if ($this->find($job->uuid)) {
                throw new QUI\Exception("Queue job with UUID '$job->uuid' already exists.");
            }

            $uuid = $job->uuid;
        } else {
            $uuid = QUI\Utils\Uuid::get();
        }

        $now = new DateTime();

        $this->db->insert(
            $this->getJobTable(),
            [
                'uuid' => $uuid,
                'jobData' => $job->data->jsonSerialize(),
                'status' => $job->status->value,
                'priority' => $job->jobAttributes->priority ?: 1,
                'deleteOnFinish' => $job->jobAttributes->deleteOnFinish ? 1 : 0,
                'createTime' => $now->format('Y-m-d H:i:s'),
                'lastUpdateTime' => $now->format('Y-m-d H:i:s'),
                'isQuiqqerWorkerJob' => $job->isQuiqqerWorkerJob ? 1 : 0,
                'earliestQueueDate' => $job->jobAttributes->earliestQueueDate?->format('Y-m-d H:i:s')
            ]
        );

        return new QueueJob(
            new QueueJobUUID($uuid),
            $job->jobAttributes,
            $now,
            $now
        );
    }

    /**
     * @param QueueJob $job
     * @return void
     * @throws QUI\Database\Exception
     * @throws QUI\Exception
     */
    public function update(QueueJob $job): void
    {
        if (is_null($job->uuid)) {
            throw new QUI\Exception("Cannot update QueueJob :: UUID is null");
        }

        $this->db->update(
            $this->getJobTable(),
            [
                'jobData' => $job->data->jsonSerialize(),
                'resultData' => $job->result?->jsonSerialize(),
                'status' => $job->status->value,
                'jobLog' => json_encode($job->log->logEntries),
                'lastUpdateTime' => date('Y-m-d H:i:s')
            ],
            [
                'uuid' => $job->uuid
            ]
        );
    }

    /**
     * @param QueueJobUUID $jobId
     * @return void
     *
     * @throws QUI\Database\Exception
     * @throws QUI\Exception
     */
    public function delete(QueueJobUUID $jobId): void
    {
        $this->db->delete(
            $this->getJobTable(),
            [
                'uuid' => $jobId->uuid
            ]
        );
    }

    /**
     * @param QueueJobUUID $jobId
     * @return QueueJob|QuiqqerWorkerQueueJob|null
     * @throws Exception
     */
    public function find(QueueJobUUID $jobId): QueueJob | QuiqqerWorkerQueueJob | null
    {
        $result = $this->db->fetch([
            'from' => $this->getJobTable(),
            'where' => [
                'uuid' => $jobId->uuid
            ],
            'limit' => 1
        ]);

        if (empty($result)) {
            return null;
        }

        return $this->parseDbRowToQueueJob($result[0]);
    }

    /**
     * @param array $where (optional) - WHERE clause (QUIQQER DB syntax)
     * @return QueueJob[]
     * @throws QUI\Database\Exception
     */
    public function findAll(array $where = []): array
    {
        $jobs = [];

        $result = $this->db->fetch([
            'from' => $this->getJobTable(),
            'where' => $where
        ]);

        foreach ($result as $row) {
            $jobs[] = $this->parseDbRowToQueueJob($row);
        }

        return $jobs;
    }

    /**
     * Get jobad list for backend job overview.
     *
     * @param array $searchParams
     * @return array
     * @throws QUI\Database\Exception
     */
    public function getListForBackendGrid(array $searchParams): array
    {
        $Grid = new Grid($searchParams);
        $gridParams = $Grid->parseDBParams($searchParams);

        $sortOn = 'id';
        $sortBy = 'ASC';

        if (!empty($searchParams['sortOn'])) {
            $sortOn = $searchParams['sortOn'];
        }

        if (!empty($searchParams['sortBy'])) {
            $sortBy = $searchParams['sortBy'];
        }

        $result = QUI::getDataBase()->fetch([
            'from' => self::getJobTable(),
            'limit' => $gridParams['limit'],
            'order' => $sortOn . ' ' . $sortBy
        ]);

        $resultCount = QUI::getDataBase()->fetch([
            'count' => 1,
            'from' => self::getJobTable()
        ]);

        $gridRows = [];

        foreach ($result as $row) {
            $job = $this->parseDbRowToQueueJob($row);

            $gridRows[] = [
                'uuid' => $job->uuid->uuid,
                'status' => $job->status->name,
                'jobWorker' => $job instanceof QUI\QueueManager\QuiqqerWorkerQueueJob ?
                    $job->getQuiqqerQueueWorkerClass() :
                    '-',
                'priority' => $job->jobAttributes->priority ?: 1,
            ];
        }

        return $Grid->parseResult(
            $gridRows,
            current(current($resultCount))
        );
    }

    /**
     * @param array $dbRow
     * @return QueueJob|QuiqqerWorkerQueueJob
     */
    private function parseDbRowToQueueJob(array $dbRow): QueueJob | QuiqqerWorkerQueueJob
    {
        $attributes = new QueueJobAttributes();

        if (isset($dbRow['priority'])) {
            $attributes->priority = (int)$dbRow['priority'];
        }

        if (isset($dbRow['deleteOnFinish'])) {
            $attributes->deleteOnFinish = (bool)$dbRow['deleteOnFinish'];
        }

        if (isset($dbRow['earliestQueueDate'])) {
            $attributes->earliestQueueDate = date_create($dbRow['earliestQueueDate']);
        }

        $data = null;
        $queueJob = null;

        if (!empty($dbRow['jobData'])) {
            $data = new JobData(json_decode($dbRow['jobData'], true));
        }

        if (!empty($dbRow['isQuiqqerWorkerJob']) && $data && !empty($data->get('workerClass'))) {
            $queueJob = new QUI\QueueManager\QuiqqerWorkerQueueJob(
                $data->get('workerClass'),
                new QueueJobUUID($dbRow['uuid']),
                $attributes
            );
        }

        if (!$queueJob) {
            $queueJob = new QueueJob(
                new QueueJobUUID($dbRow['uuid']),
                $attributes
            );
        }

        if ($data) {
            foreach ($data->getAll() as $k => $v) {
                $queueJob->data->set($k, $v);
            }
        }

        if (!empty($dbRow['jobResult'])) {
            $queueJob->result = new JobData(json_decode($dbRow['jobResult'], true));
        }

        if (!empty($dbRow['jobLog'])) {
            $queueJob->log->logEntries = json_decode($dbRow['jobLog'], true);
        }

        if (!empty($dbRow['status'])) {
            $queueJob->status = JobStatus::from((int)$dbRow['status']);
        }

        $queueJob->createDate = date_create($dbRow['createTime']);
        $queueJob->updateDate = date_create($dbRow['lastUpdateTime']);

        return $queueJob;
    }

    private function getJobTable(): string
    {
        return QUI::getDBTableName(QueueJobRepository::TBL_JOBS);
    }
}
