<?php

namespace QUI\QueueServer;

use DateTime;
use QUI;
use QUI\Exception;
use QUI\QueueManager\Exceptions\ServerException;
use QUI\QueueManager\Interfaces\QueueJobRepositoryInterface;
use QUI\QueueManager\Interfaces\QueueServerInterface;
use QUI\QueueManager\JobData;
use QUI\QueueManager\JobLog;
use QUI\QueueManager\JobStatus;
use QUI\QueueManager\QueueJob;
use QUI\QueueManager\QueueJobUUID;
use QUI\QueueManager\QuiqqerWorkerQueueJob;
use QUI\QueueManager\Repository\QueueJobRepository;

use function date;
use function is_null;

/**
 * Class QueueServer
 *
 * Quiqqer queue server (cronjob)
 *
 * @package QUI\QueueServer
 */
class Server implements QueueServerInterface
{
    /**
     * @param QueueJobRepositoryInterface|null $queueJobRepository
     */
    public function __construct(
        private ?QueueJobRepositoryInterface $queueJobRepository = null,
    ) {
        if (is_null($this->queueJobRepository)) {
            $this->queueJobRepository = new QueueJobRepository();
        }
    }

    /**
     * Adds a single job to the queue of a server
     *
     * @param QueueJob $job - The job to add to the queue
     * @return QueueJobUUID - unique Job ID
     *
     * @throws Exception
     */
    public function queueJob(QueueJob $job): QueueJobUUID
    {
        if (!($job instanceof QuiqqerWorkerQueueJob)) {
            throw new QUI\Exception(
                "quiqqer/queueserver can only queue jobs that are of type"
                . " '" . QuiqqerWorkerQueueJob::class . "'."
            );
        }

        $newQueueJob = $this->queueJobRepository->insert($job);

        return $newQueueJob->uuid;
    }

    /**
     * Get result of a specific job
     *
     * @param QueueJobUUID $jobId
     * @param bool $deleteJob (optional) - delete job from queue after return [default: true]
     * @return JobData
     *
     * @throws Exception
     */
    public function getJobResult(QueueJobUUID $jobId, bool $deleteJob = true): JobData
    {
        $job = $this->getJob($jobId);

        switch ($job->status) {
            case JobStatus::QUEUED:
                throw new QUI\Exception([
                    'quiqqer/queueserver',
                    'exception.queueserver.result.job.queued',
                    [
                        'jobId' => $jobId
                    ]
                ]);

            case JobStatus::RUNNING:
                throw new QUI\Exception([
                    'quiqqer/queueserver',
                    'exception.queueserver.result.job.running',
                    [
                        'jobId' => $jobId
                    ]
                ]);
        }

        if ($deleteJob) {
            $this->deleteJob($jobId);
        }

        return $job->result;
    }

    /**
     * Set result of a specific job
     *
     * @param QueueJobUUID $jobId
     * @param JobData $result
     * @return void
     * @throws Exception
     */
    public function setJobResult(QueueJobUUID $jobId, JobData $result): void
    {
        $job = $this->getJob($jobId);

        switch ($job->status) {
            case JobStatus::FINISHED:
                throw new QUI\Exception(
                    "Cannot set job result for $jobId :: Job is already finished."
                );

            case JobStatus::ERROR:
                throw new QUI\Exception(
                    "Cannot set job result for $jobId :: Job is in error state."
                );
        }

        $job->result = $result;
        $this->queueJobRepository->update($job);
    }

    /**
     * Delete a job
     *
     * @param QueueJobUUID $jobId
     * @return void
     * @throws Exception
     */
    public function deleteJob(QueueJobUUID $jobId): void
    {
        switch (self::getJobStatus($jobId)) {
            case self::JOB_STATUS_RUNNING:
                throw new QUI\Exception([
                    'quiqqer/queueserver',
                    'exception.queueserver.cancel.job.running',
                    [
                        'jobId' => $jobId
                    ]
                ]);
        }

        $this->queueJobRepository->delete($jobId);
    }

    /**
     * Set status of a job
     *
     * @param QueueJobUUID $jobId
     * @param JobStatus $status
     * @return void
     * @throws Exception
     */
    public function setJobStatus(QueueJobUUID $jobId, JobStatus $status): void
    {
        $job = $this->getJob($jobId);

        if ($job->status === $status) {
            return;
        }

        $job->status = $status;
        $this->queueJobRepository->update($job);
    }

    /**
     * Get status of a job
     *
     * @param QueueJobUUID $jobId
     * @return JobStatus
     * @throws Exception
     */
    public function getJobStatus(QueueJobUUID $jobId): JobStatus
    {
        $job = $this->getJob($jobId);
        return $job->status;
    }

    /**
     * Execute the next job in the queue
     *
     * @return bool - Job executed
     *
     * @throws ServerException
     * @throws Exception
     */
    public function executeNextJob(): bool
    {
        $job = $this->fetchNextJob();

        if (!$job) {
            return false;
        }

        $jobWorkerClass = $job->getQuiqqerQueueWorkerClass();

        if (!$jobWorkerClass || !class_exists($jobWorkerClass)) {
            throw new ServerException([
                'quiqqer/queueserver',
                'exception.queueserver.job.worker.not.found',
                [
                    'jobWorkerClass' => $jobWorkerClass
                ]
            ], 404);
        }

        $jobId = $job->uuid;

        try {
            /** @var QUI\QueueManager\Interfaces\QuiqqerQueueWorkerInterface $worker */
            $worker = new $jobWorkerClass();

            $this->setJobStatus($jobId, JobStatus::RUNNING);
            $jobResult = $worker->execute($job);

            $job->status = JobStatus::FINISHED;
            $job->result = $jobResult;

            $this->queueJobRepository->update($job);
        } catch (\Exception $Exception) {
            self::writeJobLogEntry($jobId, $Exception->getMessage());
            self::setJobStatus($jobId, JobStatus::ERROR);
            return false;
        }

        if ($job->jobAttributes->deleteOnFinish) {
            try {
                self::deleteJob($jobId);
            } catch (\Exception $Exception) {
                QUI\System\Log::writeException($Exception);
            }
        }

        return true;
    }

    /**
     * Fetch job data for next job in the queue (with highest priority)
     *
     * @return QuiqqerWorkerQueueJob|null
     * @throws QUI\Database\Exception
     */
    protected function fetchNextJob(): ?QuiqqerWorkerQueueJob
    {
        $sql = "SELECT `uuid` FROM `" . self::getJobTable() . "`";
        $sql .= " WHERE `status` = " . self::JOB_STATUS_QUEUED;
        $sql .= " AND (`earliestQueueDate` IS NULL OR `earliestQueueDate` <= '" . date('Y-m-d H:i:s') . "')";
        $sql .= " AND `isQuiqqerWorkerJob` = 1";
        $sql .= " ORDER BY `id` ASC, `priority` DESC";
        $sql .= " LIMIT 1";

        $result = QUI::getDataBase()->fetchSQL($sql);

        if (empty($result)) {
            return null;
        }

        /** @var QuiqqerWorkerQueueJob|null $job */
        $job = $this->queueJobRepository->find(new QueueJobUUID($result[0]['uuid']));
        return $job;
    }

    /**
     * Write log entry for a job
     *
     * @param QueueJobUUID $jobId
     * @param string $msg
     * @return void
     * @throws Exception
     */
    public function writeJobLogEntry(QueueJobUUID $jobId, string $msg): void
    {
        $job = $this->getJob($jobId);

        $job->log->addEntry([
            'time' => date('Y.m.d H:i:s'),
            'msg' => $msg
        ]);

        $this->queueJobRepository->update($job);
    }

    /**
     * Get event log for specific job
     *
     * @param QueueJobUUID $jobId
     * @return JobLog
     * @throws Exception
     */
    public function getJobLog(QueueJobUUID $jobId): JobLog
    {
        $job = $this->getJob($jobId);
        return $job->log;
    }

    /**
     * Delete all completed or failed jobs that are older than $days days
     *
     * @param int|string $days
     * @return void
     * @throws QUI\Database\Exception
     */
    public function cleanJobs(int | string $days): void
    {
        $seconds = (int)$days * 24 * 60 * 60;
        $seconds = time() - $seconds;

        QUI::getDataBase()->delete(
            self::getJobTable(),
            [
                'lastUpdateTime' => [
                    'type' => '<=',
                    'value' => $seconds
                ],
                'status' => [
                    'type' => 'IN',
                    'value' => [JobStatus::FINISHED->value, JobStatus::ERROR->value]
                ]
            ]
        );

        // OPTIMIZE
        QUI::getDataBase()->execSQL('OPTIMIZE TABLE `' . self::getJobTable() . '`');
    }

    /**
     * Get table for jobs
     *
     * @return string
     */
    public static function getJobTable(): string
    {
        return QUI::getDBTableName(QueueJobRepository::TBL_JOBS);
    }

    /**
     * Clone a job and queue it immediately
     *
     * @param QueueJobUUID $jobId - Job UUID
     * @param integer $priority - (new) job priority
     * @param DateTime|null $earliestQueueDate (optional) - Date at which the job shall be queued/executed the earliest
     *
     * @return QueueJobUUID - ID of cloned job
     * @throws Exception
     */
    public function cloneJob(QueueJobUUID $jobId, int $priority, ?DateTime $earliestQueueDate = null): QueueJobUUID
    {
        $currentJob = $this->getJob($jobId);

        if (!($currentJob instanceof QuiqqerWorkerQueueJob)) {
            throw new QUI\Exception(
                "quiqqer/queueserver can only queue jobs that are of type"
                . " '" . QuiqqerWorkerQueueJob::class . "'."
            );
        }

        $clonedJob = new QuiqqerWorkerQueueJob(
            $currentJob->getQuiqqerQueueWorkerClass(),
            null,
            $currentJob->jobAttributes
        );

        $clonedJob->data->setAll($currentJob->data->getAll());

        $clonedJob->jobAttributes->earliestQueueDate = $earliestQueueDate;
        $clonedJob->jobAttributes->priority = $priority;

        $clonedJobUuid = $this->queueJob($clonedJob);

        // Set "CLONED" status to cloned job
        $currentJob->status = JobStatus::CLONED;

        $clonedJobUuids = $currentJob->data->get(JobDataKey::CLONED_JOB_UUIDS->value);

        if (empty($clonedJobUuids)) {
            $clonedJobUuids = [];
        }

        $clonedJobUuids[] = $clonedJobUuid;
        $currentJob->data->set(JobDataKey::CLONED_JOB_UUIDS->value, $clonedJobUuids);
        $this->queueJobRepository->update($currentJob);

        return $clonedJobUuid;
    }

    /**
     * Close server connection
     */
    public function closeConnection(): void
    {
        // nothing, there is no connection that needs to be closed
    }

    /**
     * @param QueueJobUUID $jobId
     * @return QueueJob
     * @throws Exception
     */
    private function getJob(QueueJobUUID $jobId): QueueJob
    {
        $job = $this->queueJobRepository->find($jobId);

        if (is_null($job)) {
            throw new QUI\Exception("Queue job $jobId not found.");
        }

        return $job;
    }
}
