<?php

namespace QUI\Requirements;

use Exception;
use QUI;
use QUI\Requirements\Tests\Test;

use function count;
use function floor;
use function php_sapi_name;
use function strtotime;

use const PHP_EOL;

/**
 * Event Handler
 */
class EventHandler
{
    /**
     * @return void
     * @throws Exception
     */
    public static function checkImportantCron(): void
    {
        // if watcher is installed, check if watcher cleanup is installed
        $watcherInstalled = QUI::getPackageManager()->isInstalled('quiqqer/watcher');
        $output = [];

        if ($watcherInstalled) {
            if (self::isInCronList('\QUI\Watcher\Cron::clearWatcherEntries') === false) {
                $output[] = '⚠ Missing Watcher Cron - Please setup the Watcher Cron!';
            } elseif (self::lastExec('\QUI\Watcher\Cron::clearWatcherEntries') > 7) {
                $output[] = '⚠ The Watcher Cron did not run for 7 days. Please check it!';
            }
        }

        if (self::isInCronList('\QUI\Cron\QuiqqerCrons::clearSessions') === false) {
            $output[] = '⚠ Missing Session Cleanup Cron - Please setup the Session Cleanup Cron!';
        } elseif (self::lastExec('\QUI\Cron\QuiqqerCrons::clearSessions') > 7) {
            $output[] = '⚠ The Session Cleanup Cron did not run for 7 days. Please check it!';
        }

        if (self::isInCronList('\QUI\Cron\QuiqqerCrons::clearTempFolder') === false) {
            $output[] = '⚠ Missing Temp Cleanup Cron - Please setup the Temp Cleanup Cron!';
        } elseif (self::lastExec('\QUI\Cron\QuiqqerCrons::clearTempFolder') > 7) {
            $output[] = '⚠ The Temp Cleanup Cron did not run for 7 days. Please check it!';
        }

        // Check php modules
        $Req = new Requirements();
        $tests = $Req->getAllTests();

        foreach ($tests['PHP - Modules'] as $category) {
            /* @var $Test Test */
            foreach ($category as $Test) {
                $Result = $Test->getResult();

                if ($Result->getStatus() === TestResult::STATUS_OK) {
                    continue;
                }

                $message = $Result->getMessageRaw();
                $status = $Result->getStatusHumanReadable();

                if (!empty($message)) {
                    $output[] = $status . ' - ' . $message;
                } else {
                    $output[] = $status . ' - ' . $Test->getName();
                }
            }
        }


        if (count($output)) {
            self::output('#########################################################');
            self::output('#   Your quiqqer system is missing some requirements!   #');
            self::output('#########################################################');
            self::output('');
        }

        foreach ($output as $o) {
            self::output($o);
        }

        if (count($output)) {
            self::output('');
        }
    }

    /**
     * @param $msg
     * @return void
     */
    protected static function output($msg): void
    {
        // no console
        if (php_sapi_name() === 'cli') {
            echo PHP_EOL;
            echo "\033[1;31m";
            echo $msg;
            echo "\033[0m";

            return;
        }

        QUI::getMessagesHandler()->addError($msg);
    }

    /**
     * @param $exec
     * @return bool
     * @throws QUI\Database\Exception
     */
    public static function isInCronList($exec): bool
    {
        $CronManager = new QUI\Cron\Manager();
        $cronList = $CronManager->getList();

        foreach ($cronList as $cron) {
            if ($cron['exec'] === $exec && (int)$cron['active'] === 1) {
                return true;
            }
        }

        return false;
    }

    /**
     * @throws QUI\Database\Exception
     */
    public static function lastExec($exec): bool|int|float
    {
        $CronManager = new QUI\Cron\Manager();
        $cronList = $CronManager->getList();

        foreach ($cronList as $cron) {
            if ($cron['exec'] !== $exec) {
                continue;
            }

            $lastExec = $cron['lastexec'];
            $lastExec = strtotime($lastExec);

            $days = time() - $lastExec;
            $days = $days / 60 / 60 / 24;

            return floor($days);
        }

        return 9999999;
    }
}
