<?php

namespace QUI\Requirements;

use Exception;
use QUI\Requirements\Api\AbstractRequirementProvider;
use QUI\Requirements\Api\Coordinator;
use QUI\System\Log;

use function class_exists;
use function dirname;
use function get_class;
use function parse_ini_file;
use function str_replace;

/**
 * Class Locale
 *
 * @package QUI\Requirements
 */
class Locale
{
    protected string $langCode;
    protected array $locales = [];

    protected static ?Locale $Instance = null;

    /**
     * Gets the language variable for the current language
     *
     * @param $variable
     *
     * @param array $params
     *
     * @return string - The translated value
     * @throws Exception
     */
    public function get($variable, array $params = []): string
    {
        $variable = trim($variable);

        if (empty($this->langCode)) {
            $this->setlanguage("en");
        }

        if (!isset($this->locales[$this->langCode][$variable])) {
            throw new Exception(
                "Variable '" . $variable . "' not found in current language '" . $this->langCode . "'!"
            );
        }

        $text = $this->locales[$this->langCode][$variable];

        foreach ($params as $key => $value) {
            $text = str_replace("%" . $key . "%", $value, $text);
        }

        return $text;
    }

    /**
     * Sets the current language code
     *
     * @param $langCode
     *
     * @throws Exception
     */
    public function setlanguage($langCode): void
    {
        $this->loadLocales();

        if (in_array($langCode, $this->getAvailableLanguages())) {
            $this->langCode = $langCode;

            return;
        }

        // Check for the short form of the code
        if (str_contains($langCode, "_")) {
            $parts = explode("_", $langCode);

            if (in_array($parts[0], $this->getAvailableLanguages())) {
                $this->langCode = $parts[0];

                return;
            }
        }

        // Check for english as default
        if (in_array("en_GB", $this->getAvailableLanguages())) {
            $this->langCode = "en_GB";

            return;
        }

        if (in_array("en", $this->getAvailableLanguages())) {
            $this->langCode = "en";

            return;
        }

        throw new Exception("Language code is not available");
    }

    /**
     * Returns the current language code
     *
     * @return string
     */
    public function getCurrentLangugage(): string
    {
        return $this->langCode;
    }

    /**
     * Returns the available language codes
     *
     * @return array
     */
    public function getAvailableLanguages(): array
    {
        return array_keys($this->locales);
    }

    /**
     * Loads the locales and returns the locales
     *
     * @return array
     * @throws Exception
     */
    protected function loadLocales(): array
    {
        $iniFile = dirname(__FILE__, 4) . "/locales.ini.php";

        if (!file_exists($iniFile)) {
            throw new Exception("LanguageFile was not found: " . $iniFile);
        }

        $locales = parse_ini_file($iniFile, true);

        if (empty($locales)) {
            throw new Exception("No locales could be parsed!");
        }

        foreach ($locales as $lang => $values) {
            foreach ($values as $k => $value) {
                $locales[$lang][$k] = str_replace('php7.X-', Utils::getPHPCommand(), $value);
            }
        }

        $this->locales = $locales;
        $this->loadExternalModuleLocales();

        return $locales;
    }

    /**
     * Adds the locales provided by other modules to the existing locales property
     *
     * @throws Exception
     */
    protected function loadExternalModuleLocales(): void
    {
        $registeredProviders = Coordinator::getInstance()->getRequirementsProvider();

        /** @var AbstractRequirementProvider $Provider */
        foreach ($registeredProviders as $Provider) {
            try {
                $moduleLocales = $Provider->getLocales();

                foreach ($moduleLocales as $langGroup => $locales) {
                    foreach ($locales as $localeVariableName => $translation) {
                        $this->locales[$langGroup][$localeVariableName] = $translation;
                    }
                }
            } catch (Exception $Exception) {
                if (class_exists('\QUI\System\Log')) {
                    Log::addDebug(
                        'Could not load the locales for provider "' . get_class(
                            $Provider
                        ) . '": ' . $Exception->getMessage()
                    );
                }
            }
        }
    }

    /**
     * @return Locale|null
     */
    public static function getInstance(): ?Locale
    {
        if (!is_null(self::$Instance)) {
            return self::$Instance;
        }

        self::$Instance = new Locale();

        return self::$Instance;
    }
}
