<?php

namespace QUI\Requirements\Tests;

use QUI\Cache\Manager;
use QUI\Exception;
use QUI\Requirements\Locale;
use QUI\Requirements\TestResult;
use ReflectionObject;

use function defined;

/**
 * Test Abstract class
 *
 * Every test has to extend from it
 */
abstract class Test
{
    /** @var  String - Test Identifier */
    protected string $identifier;

    /** @var  TestResult TestResult */
    protected TestResult $Result;

    /** @var bool Did the test run already */
    protected bool $didRun = false;

    /**
     * @throws \Exception
     */
    public function __construct()
    {
        if (empty($this->identifier)) {
            throw new \Exception("Every test needs an identifier");
        }
    }

    /**
     * Gets the tests name
     *
     * @return string
     */
    public function getName(): string
    {
        try {
            return Locale::getInstance()->get("requirements.tests." . $this->identifier . ".name");
        } catch (\Exception) {
            return (new ReflectionObject($this))->getShortName();
        }
    }

    /**
     * Gets the tests description
     *
     * @return string
     */
    public function getDescription(): string
    {
        try {
            return Locale::getInstance()->get("requirements.tests." . $this->identifier . ".desc");
        } catch (\Exception) {
            return "";
        }
    }

    /**
     * Gets the tests description without html and php tags
     *
     * @return string
     */
    public function getDescriptionRaw(): string
    {
        try {
            $description = Locale::getInstance()->get(
                "requirements.tests." . $this->identifier . ".desc"
            );

            return strip_tags($description);
        } catch (\Exception) {
            return "";
        }
    }

    /**
     * Gets the tests group name
     *
     * @return string
     */
    public function getGroupName(): string
    {
        $ReflectionObject = new ReflectionObject($this);

        $namespace = $ReflectionObject->getNamespaceName();

        $namespace = str_replace("QUI\\Requirements\\Tests\\", "", $namespace);
        $namespace = strtolower($namespace);
        $namespace = str_replace("\\", ".", $namespace);

        try {
            return Locale::getInstance()->get("requirements.tests.groups." . $namespace);
        } catch (\Exception) {
            return ucfirst($namespace);
        }
    }

    /**
     * Returns the group's identifier.
     * The identifier is generated by the directory structure below src/QUI/Requirements/Tests.
     * The slashes get replaced by dots
     *
     * @return array|string|string[]
     */
    public function getGroupIdentifier(): array|string
    {
        $ReflectionObject = new ReflectionObject($this);

        $namespace = $ReflectionObject->getNamespaceName();
        $namespace = str_replace("QUI\\Requirements\\Tests\\", "", $namespace);
        $namespace = strtolower($namespace);

        return str_replace("\\", ".", $namespace);
    }

    /**
     * @return TestResult
     */
    public function getResult(): TestResult
    {
        if (!$this->didRun) {
            $this->Result = $this->run();

            if (defined('CMS_DIR') && class_exists('QUI\Cache\Manager')) {
                Manager::set('quiqqer.test.result.' . $this->getIdentifier(), $this->Result);
            }

            $this->didRun = true;
        }

        return $this->Result;
    }

    /**
     * Returns the last test result from the cache.
     * If the result isn't in the cache an empty TestResult with status UNKNOWN is returned.
     *
     * @return bool|array|TestResult|string
     */
    public function getResultFromCache(): bool|array|TestResult|string
    {
        try {
            return Manager::get('quiqqer.test.result.' . $this->getIdentifier());
        } catch (Exception) {
            return new TestResult(TestResult::STATUS_UNKNOWN);
        }
    }

    /**
     * @return string
     */
    public function getIdentifier(): string
    {
        return $this->identifier;
    }

    /**
     * Executes the test and returns the result
     *
     * @return TestResult
     */
    abstract protected function run(): TestResult;
}
