<?php

namespace QUI\Requirements\Tests\Webserver;

use QUI\Requirements\TestResult;
use QUI\Requirements\Tests\Test;
use QUI\System\Log;

use function parse_url;

/**
 * Class Rewrite
 *
 * @package QUI\Requirements\Tests\Webserver
 */
class Rewrite extends Test
{
    /**
     * @var string
     */
    protected string $identifier = "webserver.rewrite";

    /**
     * @return TestResult
     */
    protected function run(): TestResult
    {
        if (php_sapi_name() == "cli") {
            return new TestResult(TestResult::STATUS_UNKNOWN);
        }

        $isNginxWebserver = false;
        $isApacheWebserver = false;

        if (isset($_SERVER['SERVER_SOFTWARE']) && stripos($_SERVER['SERVER_SOFTWARE'], 'nginx') !== false) {
            $isNginxWebserver = true;
        }

        if (isset($_SERVER['SERVER_SOFTWARE']) && stripos($_SERVER['SERVER_SOFTWARE'], 'apache') !== false) {
            $isApacheWebserver = true;
        }

        if ($isNginxWebserver) {
            if ($this->testNginxRedirectByRequest()) {
                return new TestResult(TestResult::STATUS_OK);
            } else {
                return new TestResult(TestResult::STATUS_FAILED);
            }
        }

        if ($isApacheWebserver) {
            if ($this->testApacheRedirectByRequest()) {
                return new TestResult(TestResult::STATUS_OK);
            } else {
                return new TestResult(TestResult::STATUS_FAILED);
            }
        }

        return new TestResult(TestResult::STATUS_UNKNOWN);
    }

    /**
     * Executes a curl request against the server and checks if the request gets redirected properly.
     *
     * @return bool
     */
    protected function testApacheRedirectByRequest(): bool
    {
        if (php_sapi_name() == "cli") {
            return false;
        }

        if (!defined('CMS_DIR')) {
            return false;
        }

        $htaccessFile = CMS_DIR . '/.htaccess';

        $serverUrl = HOST . URL_DIR;
        $serverUrl = str_replace('https://', '', $serverUrl);

        if (!file_exists($htaccessFile)) {
            Log::addError('Missing htaccess file');
            return false;
        }

        // change htaccess file
        $originalContent = file_get_contents($htaccessFile);
        $checkValue = time();

        $htAccessContent = '# Created by rewrite health test' . PHP_EOL;
        $htAccessContent .= "RewriteEngine on" . PHP_EOL;
        $htAccessContent .= "RewriteRule ^rewritetest$ rewritetest.html [NC]" . PHP_EOL;
        $htAccessContent .= PHP_EOL;

        file_put_contents($htaccessFile, $htAccessContent . $originalContent);
        file_put_contents(CMS_DIR . "/rewritetest.html", $checkValue);

        // HTTP TEST
        $ch = curl_init('http://' . $serverUrl . "/rewritetest");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

        $http_result = curl_exec($ch);
        curl_close($ch);


        // HTTPS TEST
        $ch = curl_init('https://' . $serverUrl . "/rewritetest");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

        $https_result = curl_exec($ch);
        curl_close($ch);


        unlink(CMS_DIR . "/rewritetest.html");
        file_put_contents($htaccessFile, $originalContent);

        if ($http_result == $checkValue || $https_result == $checkValue) {
            return true;
        }

        return false;
    }

    /**
     * Test nginx rewrite functionality
     *
     * @return bool
     * @todo: must be implemented
     *
     */
    protected function testNginxRedirectByRequest(): bool
    {
        return true;
    }

    /**
     * @param $url
     *
     * @return string
     *
     * found at https://stackoverflow.com/a/29147968
     * thx!
     */
    protected function currentDir($url): string
    {
        // note: anything without a scheme ("example.com", "example.com:80/", etc.) is a folder
        // remove query (protection against "?url=http://example.com/")
        if ($first_query = strpos($url, '?')) {
            $url = substr($url, 0, $first_query);
        }
        // remove fragment (protection against "#http://example.com/")
        if ($first_fragment = strpos($url, '#')) {
            $url = substr($url, 0, $first_fragment);
        }
        // folder only
        $last_slash = strrpos($url, '/');

        if (!$last_slash) {
            return '/';
        }
        // add ending slash to "http://example.com"
        if (($first_colon = strpos($url, '://')) !== false && $first_colon + 2 == $last_slash) {
            return $url . '/';
        }

        return substr($url, 0, $last_slash + 1);
    }
}
