<?php

/**
 * This file contains QUI\Slider\Controls\ScrollingStrip
 */

namespace QUI\Slider\Controls;

use QUI;
use QUI\Projects\Media\Folder;

/**
 * Class ScrollingStrip
 *
 * @package QUI\Slider\Controls
 * @author  www.pcsg.de (Michael Danielczok)
 */
class ScrollingStrip extends QUI\Control
{
    /**
     * @param array $attributes
     */
    public function __construct(array $attributes = [])
    {
        // default options
        $this->setAttributes([
            'class' => 'quiqqer-slider-controls-scrollingStrip',
            'nodeName' => 'section',
            'folderId' => null,
            'order' => 'random',
            'entries' => [], // array or valid json string
            'verticalAlign' => 'stretch', // stretch / flex-start / center / flex-end
            'animationDirection' => 'toLeft', // toLeft / toRight
            'animationDuration' => 60, // in seconds
            'pauseOnHover' => true,
            'dimItemsUntilHover' => false, // dim (grayscale and opacity) items until hover
            'imageMaxHeight' => 80 // in px
        ]);

        $this->addCSSFile(
            dirname(__FILE__) . '/ScrollingStrip.css'
        );

        parent::__construct($attributes);
    }

    /**
     * (non-PHPdoc)
     *
     * @see \QUI\Control::create()
     */
    public function getBody(): string
    {
        $Engine = QUI::getTemplateManager()->getEngine();
        $folderId = $this->getAttribute('folderId');
        $images = [];

        if ($folderId) {
            $images = $this->getImages();
        }

        $entries = $this->getAttribute('entries');

        if (is_string($entries)) {
            $entries = json_decode($entries, true);

            if (!is_array($entries)) {
                QUI\System\Log::addWarning('Invalid JSON entries for ScrollingStrip Control', [
                    'class' => static::class,
                    'method' => __METHOD__,
                    'rawEntries' => $this->getAttribute('entries')
                ]);

                $entries = [];
            }
        } elseif (!is_array($entries)) {
            $entries = [];
        }

        if (empty($entries)) {
            QUI\System\Log::addWarning('No entries found for ScrollingStrip Control', [
                'class' => static::class,
                'method' => __METHOD__,
                'file' => __FILE__,
                'line' => __LINE__
            ]);

            return '';
        }

        $imageMaxHeight = $this->getAttribute('imageMaxHeight');

        if (!$imageMaxHeight || $imageMaxHeight <= 0) {
            $imageMaxHeight = 80;
        }

        $verticalAlign = match ($this->getAttribute('verticalAlign')) {
            'stretch', 'flex-start', 'center', 'flex-end' => $this->getAttribute('verticalAlign'),
            default => 'stretch'
        };


        if ($this->getAttribute('animationDuration') && $this->getAttribute('animationDuration') > 0) {
            $this->setCustomVariable('animationDuration', $this->getAttribute('animationDuration') . 's');
        }

        $this->setCustomVariable('imageMaxHeight', $imageMaxHeight . 'px');
        $this->setCustomVariable('verticalAlign', $verticalAlign);

        $Engine->assign([
            'this' => $this,
            'images' => $images,
            'entries' => $entries,
            'animationDirection' => $this->getAttribute('animationDirection'),
            'animationDuration' => $this->getAttribute('animationDuration'),
            'pauseOnHover' => $this->getAttribute('pauseOnHover'),
            'dimItemsUntilHover' => $this->getAttribute('dimItemsUntilHover'),
            'imageMaxHeight' => $imageMaxHeight
        ]);

        return $Engine->fetch(dirname(__FILE__) . '/ScrollingStrip.html');
    }

    /**
     * Set custom css variable to the control as inline style
     * --_qui-settings-$name: var(--qui-slider-controls-scrollingStrip-$name, $value);
     *
     * Example:
     *     --_qui-settings-verticalAlign: var(--qui-slider-controls-scrollingStrip-verticalAlign, center);
     *
     * @param string $name
     * @param string $value
     *
     * @return void
     */
    private function setCustomVariable(string $name, string $value): void
    {
        if (!$name || !$value) {
            return;
        }

        $this->setStyle(
            '--_qui-settings-' . $name,
            'var(--qui-slider-controls-scrollingStrip-' . $name . ', ' . $value . ')'
        );
    }

    /**
     * Get images from folder.
     *
     * If the folder cannot be resolved (e.g. invalid ID/URL or missing permissions),
     * an empty array is returned.
     *
     * @return array
     */
    private function getImages(): array
    {
        $folderId = $this->getAttribute('folderId');
        /** @var Folder|null $Folder */
        $Folder = null;

        if (is_string($folderId) && str_contains($folderId, 'image.php')) {
            try {
                $Folder = QUI\Projects\Media\Utils::getMediaItemByUrl(
                    $folderId
                );
            } catch (QUI\Exception) {
            }
        } elseif (!empty($folderId)) {
            try {
                $Project = $this->getProject();
                $Media = $Project->getMedia();
                $Folder = $Media->get((int)$folderId);
            } catch (QUI\Exception) {
            }
        }

        switch ($this->getAttribute('order')) {
            case 'title DESC':
            case 'title ASC':
            case 'name DESC':
            case 'name ASC':
            case 'c_date DESC':
            case 'c_date ASC':
            case 'e_date DESC':
            case 'e_date ASC':
            case 'priority DESC':
            case 'priority ASC':
                $order = $this->getAttribute('order');
                break;

            default:
                $order = 'name DESC';
                break;
        }

        $images = [];

        if ($Folder && method_exists($Folder, 'getImages')) {
            $images = $Folder->getImages([
                'order' => $order
            ]);
        }

        return $images;
    }
}
