<?php

/**
 * This file contains QUI\Slider\Controls\Slider
 */

namespace QUI\Slider\Controls;

use QUI;
use QUI\Projects\Media\Utils;
use QUI\Projects\Site\Utils as SiteUtils;

/**
 * Class Slider
 */
class Slider extends QUI\Control
{
    /**
     * Image data list
     *
     * @var array<int, mixed>
     */
    protected array $data = [];

    /**
     * internal image parsing flag
     *
     * @var bool
     */
    protected bool $imagesParsing = false;

    /**
     * @var array<string>
     */
    protected array $settings = [];

    /**
     * constructor
     *
     * @param array<string, mixed> $attributes
     */
    public function __construct(array $attributes = [])
    {
        // default options
        $this->setAttributes([
            'title' => '',
            'text' => '',
            'class' => 'quiqqer-slider',
            'nodeName' => 'section',
            'qui-class' => 'package/quiqqer/slider/bin/Slider'
        ]);

        $this->addCSSFile(
            dirname(__FILE__) . '/Slider.css'
        );

        $this->addCSSFile(
            OPT_DIR . 'quiqqer/gallery/lib/QUI/Gallery/Controls/Slider.css'
        );

        $this->settings = [
            'autostart',
            'shadow',
            'showControlsAlways',
            'showTitleAlways',
            'period',
            'type'
        ];


        parent::__construct($attributes);

        $this->setAttribute('type', 'standard');
    }

    public function getBody(): string
    {
        $Engine = QUI::getTemplateManager()->getEngine();

        if (!$this->imagesParsing && $this->getAttribute('images')) {
            $images = json_decode($this->getAttribute('images'), true);

            foreach ($images as $image) {
                $this->addImage(
                    $image['image'],
                    $image['link'],
                    $image['text']
                );
            }
        }

        foreach ($this->settings as $setting) {
            $this->setAttribute('data-' . $setting, $this->getAttribute($setting));
        }

        $Engine->assign([
            'this' => $this
        ]);


        return $Engine->fetch(dirname(__FILE__) . '/Slider.html');
    }

    /**
     * Add an image to the slider
     *
     * @param string $imagePath
     * @param bool|string $link
     * @param bool|string $text
     * @param bool|string $target
     */
    public function addImage(
        string $imagePath,
        bool | string $link = false,
        bool | string $text = false,
        bool | string $target = false
    ): void {
        try {
            $Image = Utils::getImageByUrl($imagePath);
        } catch (QUI\Exception) {
            return;
        }

        if (!$Image->isActive()) {
            return;
        }

        try {
            if (SiteUtils::isSiteLink((string)$link)) {
                $link = SiteUtils::rewriteSiteLink((string)$link);
            }
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::addDebug('##################');
            QUI\System\Log::addDebug((string)$link);
            QUI\System\Log::addDebug($Exception->getMessage());

            if (!is_string($link)) {
                $link = false;
            }
        }

        $this->data[] = [
            'link' => $link,
            'text' => $text,
            'image' => $Image,
            'target' => $target
        ];
    }

    /**
     * Return the slider image data entries
     *
     * @return array<int, mixed>
     */
    public function getList(): array
    {
        return $this->data;
    }

    /**
     * Return only the slider images
     *
     * @return array<int, QUI\Projects\Media\Image>
     */
    public function getImages(): array
    {
        $images = [];

        foreach ($this->data as $data) {
            $images[] = $data['image'];
        }

        return $images;
    }
}
