<?php

/**
 * This file contains QUI\Smarty\Smarty4
 */

namespace QUI\Smarty;

use Exception;
use QUI;
use QUI\Projects\Site\Canonical;
use Smarty;
use SmartyException;

use function class_exists;
use function dirname;
use function file_exists;
use function is_string;
use function md5;
use function method_exists;
use function str_replace;

/**
 * Smarty3 Interface for QUIQQER
 * provides the Smarty3 engine for QUIQQER
 *
 * @author  www.pcsg.de (Henning Leutz)
 * @package com.pcsg.quiqqer.package.smarty3
 */
class Smarty4 implements QUI\Interfaces\Template\EngineInterface
{
    /**
     * @var array|null
     */
    public static ?array $fileCache = null;

    /**
     * @var null|Smarty
     */
    protected ?Smarty $Smarty = null;

    /**
     * @var QUI\Locale|null
     */
    protected ?QUI\Locale $Locale = null;

    /**
     * @var QUI\Projects\Project|null
     */
    protected ?QUI\Projects\Project $Project = null;

    /**
     * @var QUI\Package\Package|null
     */
    protected ?QUI\Package\Package $TemplatePackage = null;

    /**
     * @var ?QUI\Package\Package
     */
    protected null | QUI\Package\Package $TemplateParent = null;

    /**
     * construct
     *
     * @param boolean $admin
     * @throws SmartyException|QUI\Exception
     */
    public function __construct(bool $admin = false)
    {
        // Templates
        QUI\Utils\System\File::mkdir(VAR_DIR . 'cache/templates');
        QUI\Utils\System\File::mkdir(VAR_DIR . 'cache/compile');
        QUI\Utils\System\File::mkdir(VAR_DIR . 'cache/cache');

        if (!class_exists('\Smarty_Autoloader')) {
            require OPT_DIR . 'smarty/smarty/libs/bootstrap.php';
        }

        $Smarty = new Smarty();
        $Smarty->setTemplateDir(VAR_DIR . 'cache/templates');
        $Smarty->setCompileDir(VAR_DIR . 'cache/compile');
        $Smarty->setCacheDir(VAR_DIR . 'cache/cache');

        $Smarty->compile_check = Smarty::CACHING_OFF;

        if (QUI::conf('smarty', 'compile_check')) {
            $Smarty->compile_check = Smarty::COMPILECHECK_ON;
        }

        $noCache = QUI\Cache\Manager::getConfig()->get('general', 'nocache');

        if (!empty($noCache)) {
            $Smarty->compile_check = Smarty::CACHING_OFF;
            $Smarty->caching = Smarty::CACHING_OFF;
            $Smarty->clearCompiledTemplate();
        }

        $DIR = dirname(__FILE__);

        $plugin_dir = $Smarty->getPluginsDir();
        $plugin_dir[] = $DIR . '/plugins_qui/';

        if ($admin) {
            $plugin_dir[] = $DIR . '/plugins_qui_admin/';
        }

        $Smarty->setPluginsDir($plugin_dir);

        /**
         * This overwrites the default smarty "fetch" method!
         */
        $Smarty->registerPlugin(
            "function",
            "fetch",
            "QUI\Smarty\plugins_qui\QuiqqerTemplateFetch::fetch"
        );

        $Smarty->registerClass('QUI', '\QUI');

        $Smarty->registerPlugin('modifier', 'md5', 'md5');
        $Smarty->registerPlugin('modifier', 'strpos', 'strpos');
        $Smarty->registerPlugin('modifier', 'defined', 'defined');
        $Smarty->registerPlugin('modifier', 'is_string', 'is_string');
        $Smarty->registerPlugin('modifier', 'get_class', 'get_class');
        $Smarty->registerPlugin('modifier', 'uasort', 'uasort');
        $Smarty->registerPlugin('modifier', 'method_exists', 'method_exists');
        $Smarty->registerPlugin('modifier', 'trim', 'trim');

        $Smarty->registerPlugin('function', 'md5', 'md5');
        $Smarty->registerPlugin('function', 'strpos', 'strpos');
        $Smarty->registerPlugin('function', 'defined', 'defined');
        $Smarty->registerPlugin('function', 'is_string', 'is_string');
        $Smarty->registerPlugin('function', 'get_class', 'get_class');
        $Smarty->registerPlugin('function', 'uasort', 'uasort');
        $Smarty->registerPlugin('function', 'method_exists', 'method_exists');
        $Smarty->registerPlugin('function', 'trim', 'trim');

        $Smarty->registerPlugin('function', 'strftime', '\PHP81_BC\strftime');
        $Smarty->registerPlugin('modifier', 'strftime', '\PHP81_BC\strftime');

        try {
            QUI::getEvents()->fireEvent('smartyInit', [$Smarty]);
        } catch (QUI\ExceptionStack $Exception) {
            $list = $Exception->getExceptionList();

            /* @var $Exc QUI\Exception */
            foreach ($list as $Exc) {
                QUI\System\Log::writeException($Exc);
            }
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::writeException($Exception);
        }

        $this->Smarty = $Smarty;
        $this->Project = QUI::getRewrite()->getProject();

        if (!$this->Project) {
            $this->Project = QUI::getProjectManager()->getStandard();
        }

        try {
            $this->TemplatePackage = QUI::getPackage($this->Project->getAttribute('template'));

            if (method_exists($this->TemplatePackage, 'getTemplateParent')) {
                $this->TemplateParent = $this->TemplatePackage->getTemplateParent();
            }
        } catch (QUI\Exception) {
        }

        if (QUI::getRewrite()->getSite()) {
            $Site = QUI::getRewrite()->getSite();

            try {
                $ownTemplate = $Site->getAttribute('quiqqer.site.template');
                $Package = QUI::getPackage($ownTemplate);
                $this->TemplatePackage = $Package;

                if (method_exists($this->TemplatePackage, 'getTemplateParent')) {
                    $this->TemplateParent = $this->TemplatePackage->getTemplateParent();
                }
            } catch (QUI\Exception) {
            }
        }
    }


    public function getSmarty(): ?Smarty
    {
        return $this->Smarty;
    }

    /**
     * Return the canonical object
     *
     * @return Canonical
     */
    public function getCanonical(): QUI\Projects\Site\Canonical
    {
        return $this->getTemplateVariable('Canonical');
    }

    /**
     * @param string $variableName
     * @return mixed
     * @see QUI\Interfaces\Template::getTemplateVariable()
     *
     */
    public function getTemplateVariable(string $variableName): mixed
    {
        return $this->Smarty->getTemplateVars($variableName);
    }

    /**
     * (non-PHPdoc)
     *
     * @param string|array $var
     * @param Boolean $value
     * @see QUI\Interfaces\Template::assign()
     *
     */
    public function assign(string | array $var, $value = false): void
    {
        if (is_string($var)) {
            $this->Smarty->assign($var, $value);

            return;
        }

        $this->Smarty->assign($var);
    }

    /**
     * executes & returns or displays the template results
     *
     * @param string $resource_name
     * @return string
     */
    public function fetch(string $resource_name): string
    {
        $error = QUI::getErrorHandler()->getAttribute('ERROR_2');

        // Error Behandlung bei Smarty ausschalten, zuviele fehler
        QUI::getErrorHandler()->setAttribute('ERROR_2', false);

        // exist a usr template?
        // template ist überschreibbar im usr template
        $Project = $this->Project;
        $projectName = $Project->getName();
        $usr_resource_name = false;
        $tpl_resource_name = false;
        $template = $Project->getAttribute('template');
        $cacheManager = 'quiqqer/template/smarty/engine/fetch';

        if (self::$fileCache === null) {
            try {
                self::$fileCache = QUI\Cache\Manager::get($cacheManager);
            } catch (QUI\Exception) {
                self::$fileCache = [];
            }
        }

        $cacheName = md5($resource_name . '_' . $projectName);

        if (isset(self::$fileCache[$cacheName])) {
            $usr_resource_name = self::$fileCache[$cacheName];
        } elseif (str_contains($resource_name, OPT_DIR)) {
            $usr_resource_name = str_replace(
                OPT_DIR,
                USR_DIR . $projectName . '/lib/',
                $resource_name
            );

            // template inheritance
            $parent_resource_name = false;

            if (!$this->TemplateParent) {
                $tpl_resource_name = str_replace(
                    OPT_DIR,
                    OPT_DIR . $template . '/',
                    $resource_name
                );
                // inheritance
            } else {
                $parentPath = OPT_DIR . $this->TemplateParent->getName() . '/';
                $templatePath = OPT_DIR . $this->TemplatePackage->getName() . '/';

                if (!str_contains($resource_name, $parentPath)) {
                    // default template
                    $tpl_resource_name = str_replace(
                        OPT_DIR,
                        $templatePath,
                        $resource_name
                    );

                    $parent_resource_name = str_replace(
                        OPT_DIR,
                        $parentPath,
                        $resource_name
                    );
                } else {
                    // template from parent template
                    // we have to use the template paths here
                    $tpl_resource_name = str_replace(
                        $parentPath,
                        $templatePath,
                        $resource_name
                    );

                    $parent_resource_name = str_replace(
                        $templatePath,
                        $parentPath,
                        $resource_name
                    );
                }
            }

            if (file_exists($usr_resource_name)) {
                self::$fileCache[$cacheName] = $usr_resource_name;
                $tpl_resource_name = false;
            } elseif (file_exists($tpl_resource_name)) {
                self::$fileCache[$cacheName] = $tpl_resource_name;
                $usr_resource_name = false;
            } elseif ($parent_resource_name && file_exists($parent_resource_name)) {
                // inheritance -> parent templates
                self::$fileCache[$cacheName] = $parent_resource_name;
                $tpl_resource_name = $parent_resource_name;
                $usr_resource_name = false;
            } else {
                self::$fileCache[$cacheName] = $resource_name;
                $usr_resource_name = false;
                $tpl_resource_name = false;
            }

            QUI\Cache\Manager::set($cacheManager, self::$fileCache);
        } elseif (str_contains($resource_name, LIB_DIR)) {
            $usr_resource_name = str_replace(
                LIB_DIR,
                USR_DIR . $projectName . '/lib/',
                $resource_name
            );

            $tpl_resource_name = str_replace(
                LIB_DIR,
                OPT_DIR . $template . '/',
                $resource_name
            );

            if (file_exists($usr_resource_name)) {
                self::$fileCache[$cacheName] = $usr_resource_name;
                $tpl_resource_name = false;
            } elseif (file_exists($tpl_resource_name)) {
                self::$fileCache[$cacheName] = $tpl_resource_name;
                $usr_resource_name = false;
            } else {
                self::$fileCache[$cacheName] = $resource_name;
                $usr_resource_name = false;
                $tpl_resource_name = false;
            }

            QUI\Cache\Manager::set($cacheManager, self::$fileCache);
        }

        if ($usr_resource_name) {
            $resource_name = $usr_resource_name;
        }

        if ($tpl_resource_name) {
            $resource_name = $tpl_resource_name;
        }

        QUI\System\Log::addDebug('Engine Template -> ' . $resource_name);

        try {
            $this->Smarty->assign([
                'QUIQQER_LOCALE' => $this->Locale
            ]);

            $tpl = $this->Smarty->fetch($resource_name);
        } catch (Exception $Exception) {
            QUI\System\Log::writeRecursive($Exception->getMessage());
            QUI\System\Log::writeRecursive($Exception->getTraceAsString());

            return '';
        }

        // Errors wieder einschalten, falls es aus war
        QUI::getErrorHandler()->setAttribute('ERROR_2', $error);

        return $tpl;
    }

    /**
     * Set Project for Smarty3 engine
     *
     * @param QUI\Projects\Project $Project
     * @return void
     */
    public function setProject(QUI\Projects\Project $Project): void
    {
        $this->Project = $Project;
    }

    //region locale

    /**
     * Set the locale for the engine
     *
     * @param QUI\Locale $Locale
     */
    public function setLocale(QUI\Locale $Locale): void
    {
        $this->Locale = $Locale;
    }

    /**
     * Return the engine locale
     *
     * @return QUI\Locale|null
     */
    public function getLocale(): ?QUI\Locale
    {
        return $this->Locale;
    }

    //endregion
}
