<?php

/**
 * Smarty plugin
 *
 * @package    com.pcsg.pms.smarty
 * @subpackage plugins
 */

use ForceUTF8\Encoding;
use QUI\Projects\Media\ExternalImage;
use QUI\Projects\Media\Folder;
use QUI\Projects\Media\Image;
use QUI\Projects\Media\Video;

/**
 * Smarty {image} function plugin
 *
 * @param array<string, mixed> $params
 * @param Smarty | Smarty_Internal_Template $smarty
 * @return string
 *
 * @throws Exception
 *
 * @event onSmartyImageBegin [array &$params]
 * @event onSmartyImageEnd [string &$imageString]
 * @event onSmartyImageBeforeSource [string &$src]
 */
function smarty_function_image(array $params, Smarty | Smarty_Internal_Template $smarty): string
{
    // defaults
    if (!isset($params['type'])) {
        $params['type'] = 'resize';
    }

    try {
        QUI::getEvents()->fireEvent('smartyImageBegin', [$smarty, &$params]);
    } catch (QUI\Exception $Exception) {
        QUI\System\Log::writeDebugException($Exception);
    }

    if (!isset($params['image'])) {
        if (empty($params['src'])) {
            $src = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAABc0lEQVR4XqWTMUsDQ' .
                'RCFJyjYSFKkSS9XCGKTCEFQNglip1VSpFG0sLGxSSeeaCEp7K2sbOJPkER7EwvBKmid5porBCMyzpvlhjNoYwJf' .
                '3sybmb3NhMswM03zmVU52ICEwolQER7+6HfCvXCq/Vd3RLjB134lFJg7FywKHPw08CZ6Qvha/NhZ0wLfnKlKDuw' .
                'QxL/12AHvzbITmK+PgTYgV98DL6lbDbO6xLhetN+XrdVIP/k8xZ2OhtlGgyiKNI67XUr2lL0d+BtEW0uJOoH58h' .
                'DgqcByqQGXntGv0WYATWInMJ/v/UA8gOH0jP8bP8djiGJxLkc0Gvm4UDB/sk9PeVstJOoEf+2j7TTqSQ24ZMaWO' .
                'FzJ2xKDVouo10NOw35fNSiVVKlapWG7bUsMHiN/g5fleSf4J++uK8jV95gvPVazJT4vzvnhZlmRHNjCEAtWRy9y' .
                'O+ApmAkF5nqRRYGTmNLAm+gJbQeDhcy/XqbiK9PUr/M3T3GnVEK0IY4AAAAASUVORK5CYII=';

            // Get the placeholder if available
            try {
                $CurrentProject = QUI::getRewrite()->getProject();
            } catch (QUI\Exception) {
                $src = '<img src="' . $src . '" class="quiqqer-empty-image" />';

                return smarty_plugin_image_assign($params, $src, $smarty);
            }

            if (!$CurrentProject instanceof QUI\Projects\Project) {
                $src = '<img src="' . $src . '" class="quiqqer-empty-image" />';

                return smarty_plugin_image_assign($params, $src, $smarty);
            }

            $PlaceHolder = $CurrentProject->getMedia()->getPlaceholderImage();

            if ($PlaceHolder) {
                $src = $PlaceHolder->getSizeCacheUrl();
            }

            if (isset($params['onlyicon'])) {
                return smarty_plugin_image_assign($params, '', $smarty);
            }

            if (isset($params['onlysrc'])) {
                return smarty_plugin_image_assign($params, $src, $smarty);
            }

            $src = '<img src="' . $src . '" class="quiqqer-empty-image" />';

            return smarty_plugin_image_assign($params, $src, $smarty);
        }

        // Image Params
        if (!str_contains($params['src'], 'image.php')) {
            $onlySrcCheck = (isset($params['onlyicon']) || !isset($params['onlysrc']) || !$params['onlysrc']);
            $IconHandler = QUI\Icons\Handler::getInstance();

            if (
                $onlySrcCheck
                && $IconHandler
                && $IconHandler->isIcon((string)$params['src'])
            ) {
                return smarty_plugin_image_assign(
                    $params,
                    '<span class="quiqqer-icon ' . $params['src'] . '"></span>',
                    $smarty
                );
            }

            if (str_contains($params['src'], 'fa fa-')) {
                return smarty_plugin_image_assign($params, '', $smarty);
            }

            QUI\System\Log::writeRecursive(
                'unknown image',
                QUI\System\Log::LEVEL_WARNING,
                [
                    'smarty' => '{image}',
                    'image' => $params['src']
                ]
            );

            return smarty_plugin_image_assign($params, '', $smarty);
        }

        if (isset($params['onlyicon'])) {
            return smarty_plugin_image_assign($params, '', $smarty);
        }

        $src = str_replace('&amp;', '&', $params['src']);
        $attributes = QUI\Utils\StringHelper::getUrlAttributes($src);

        if (!isset($attributes['id']) || !isset($attributes['project'])) {
            if (isset($params['onlysrc'])) {
                return smarty_plugin_image_assign(
                    $params,
                    $params['src'],
                    $smarty
                );
            }

            return smarty_plugin_image_assign($params, '', $smarty);
        }

        try {
            $Project = QUI::getProject($attributes['project']);
            $Media = $Project->getMedia();

            /* @param $Image Image */
            $Image = $Media->get((int)$attributes['id']);
        } catch (Exception $Exception) {
            QUI\System\Log::addNotice(
                'Smarty Image Plugin {image} : ' . $Exception->getMessage()
            );

            if (isset($params['onlysrc'])) {
                return smarty_plugin_image_assign(
                    $params,
                    $params['src'],
                    $smarty
                );
            }

            return smarty_plugin_image_assign($params, '', $smarty);
        }
    } else {
        $Image = $params['image'];

        // Falls $Image ein Folder ist, dann das erste Bild nehmen
        if (QUI\Projects\Media\Utils::isFolder($Image)) {
            /* @var $Image Folder */
            try {
                $Image = $Image->firstImage();
            } catch (Exception) {
                $Image = false;
            }
        }
    }

    if (!$Image) {
        return smarty_plugin_image_assign($params, '', $smarty);
    }

    /* @var $Image Image */
    // Falls das Objekt gewünscht ist
    if (isset($params['assign']) && isset($params['object'])) {
        $smarty->assign($params['assign'], $Image);

        return '';
    }

    $mimeType = $Image->getAttribute('mime_type');

    if ($Image->getType() === Video::class) {
        $html = '';

        if (str_starts_with($mimeType, 'video/')) {
            $width = '';
            $height = '';
            $controls = '';
            $autoplay = '';
            $loop = '';
            $muted = '';
            $playsinline = '';
            $style = '';

            if (isset($params['width'])) {
                $width = $params['width'];
            }

            if (isset($params['height'])) {
                $height = $params['height'];
            }

            if (!empty($params['controls'])) {
                $controls = 'controls';
            }

            if (!empty($params['autoplay'])) {
                $autoplay = 'autoplay=""';
            }

            if (!empty($params['loop'])) {
                $loop = 'loop=""';
            }

            if (!empty($params['muted'])) {
                $muted = 'muted=""';
            }

            if (!empty($params['playsinline'])) {
                $playsinline = 'playsinline=""';
            }

            if (!empty($params['style'])) {
                $style = 'style="' . $params['style'] . '"';
            }

            $src = $Image->getUrl(true);
            $type = '';

            if (strpos($src, '.mp4')) {
                $type = 'video/mp4';
            }

            if (strpos($src, '.ogg')) {
                $type = 'video/ogg';
            }

            $html = '
                <video width="' . $width . '" height="' . $height . '" ' . $controls . ' ' . $autoplay . ' ' . $loop . ' ' . $muted . ' ' . $playsinline . ' ' . $style . ' >
                  <source src="' . $src . '" type="' . $type . '" />
                    Your browser does not support the video tag.
                </video>
            ';
        }

        return smarty_plugin_image_assign($params, $html, $smarty);
    }

    if (!isset($params['height'])) {
        $params['height'] = false;
    }

    if (!isset($params['width'])) {
        $params['width'] = false;
    }

    $maxWidth = '';
    $maxHeight = '';
    $ratio = false;

    try {
        $maxWidth = $Image->getWidth();
        $maxHeight = $Image->getHeight();

        if ($maxWidth && $maxHeight) {
            $ratio = $maxWidth / $maxHeight;
        }
    } catch (Exception $Exception) {
        QUI\System\Log::addDebug($Exception->getMessage());
    }

    // if no width and height is passed in, and we have a max width and height,
    // so we always set the original width and height
    // exception: SVG
    $isSVG = ($mimeType === 'image/svg+xml' || $mimeType === 'image/svg');

    if (!$isSVG && !$params['width'] && !$params['height'] && $maxWidth && $maxHeight) {
        $params['width'] = $maxWidth;
        $params['height'] = $maxHeight;
    }

    // make sure the height and width are not greater than in original image
    if (!$isSVG && $maxWidth && $params['width'] > $maxWidth) {
        $params['width'] = $maxWidth;
    }

    if (!$isSVG && $maxHeight && $params['height'] > $maxHeight) {
        $params['height'] = $maxHeight;
    }

    // calc correct width or height if one of them is not passed in
    // 1. rounds a number up - only integer is allowed as html width and height attribute
    if ($ratio && empty($params['width']) && !empty($params['height'])) {
        $params['width'] = ceil($params['height'] * $ratio); // (1)
    }

    if ($ratio && empty($params['height']) && !empty($params['width'])) {
        $params['height'] = ceil($params['width'] / $ratio); // (1)
    }

    switch ($params['type']) {
        default:
        case 'resize':
            try {
                $src = $Image->createResizeCache($params['width'], $params['height']);
                $resizeData = $Image->getResizeSize($params['width'], $params['height']);

                if ($resizeData['width'] && $resizeData['width'] < $params['width']) {
                    $params['width'] = $resizeData['width'];
                }
            } catch (Exception) {
                if (isset($params['onlysrc']) && isset($params['src'])) {
                    return smarty_plugin_image_assign(
                        $params,
                        $params['src'],
                        $smarty
                    );
                }

                return smarty_plugin_image_assign(
                    $params,
                    '',
                    $smarty
                );
            }
            break;
    }

    // Convert SVG to PNG
    if (!empty($params['svgtopng']) && str_contains($src, '.svg')) {
        $pngImage = $src . '.png';

        if (file_exists($pngImage)) {
            $src = $pngImage;
        } elseif (class_exists('\Imagick')) {
            $svg = file_get_contents($src);

            if ($svg === false) {
                return smarty_plugin_image_assign($params, '', $smarty);
            }

            try {
                $im = new Imagick();
                $im->readImageBlob($svg);
                $im->setImageBackgroundColor(new ImagickPixel('transparent'));
                $im->setImageFormat("png24");
                $im->writeImage($pngImage);
                $im->clear();
                $im->destroy();

                $src = $pngImage;
            } catch (Exception $Exception) {
                QUI\System\Log::addError($Exception->getMessage());
                QUI\System\Log::addDebug($Exception->getTraceAsString());

                return smarty_plugin_image_assign($params, '', $smarty);
            }
        }

        $src = str_replace(CMS_DIR, URL_DIR, $src);

        if (isset($params['onlyicon'])) {
            return smarty_plugin_image_assign($params, '', $smarty);
        }

        if (isset($params['onlysrc'])) {
            return smarty_plugin_image_assign($params, $src, $smarty);
        }

        $src = '<img src="' . $src . '"';

        if (!empty($params['width'])) {
            $src .= ' width="' . (int)$params['width'] . '"';
        }

        if (!empty($params['height'])) {
            $src .= ' height="' . (int)$params['height'] . '"';
        }

        $src .= '/>';

        return smarty_plugin_image_assign($params, $src, $smarty);
    }

    $src = str_replace(CMS_DIR, URL_DIR, $src);
    $host = false;

    if (isset($params['host']) && $params['host'] == 1) {
        $host = $Image->getMedia()->getProject()->getVHost(true, true);
        $src = $host . $src;
    }

    if (isset($params['onlysrc'])) {
        QUI::getEvents()->fireEvent('smartyImageOnlySource', [$smarty, &$src, &$params]);

        return smarty_plugin_image_assign($params, $src, $smarty);
    }


    // create picture / image tag

    try {
        QUI::getEvents()->fireEvent('smartyImageBeforeSource', [$smarty, &$src, &$params]);
    } catch (QUI\Exception $Exception) {
        QUI\System\Log::writeDebugException($Exception);
    }

    $attributes = [];
    $filter = array_flip([
        'src',
        'type',
        'reflection',
        'image',
        'assign',
        'host',
        'nosrcset'
    ]);

    foreach ($params as $key => $value) {
        if (!isset($filter[$key])) {
            $attributes[$key] = $value;
        }
    }


    // missing alt / title
    if (!isset($attributes['alt']) && is_object($Image) && method_exists($Image, 'getAlt')) {
        $attributes['alt'] = Encoding::toUTF8($Image->getAlt());
    }

    if (!isset($attributes['title']) && is_object($Image) && method_exists($Image, 'getTitle')) {
        $attributes['title'] = Encoding::toUTF8($Image->getTitle());
    }

    if ($Image instanceof ExternalImage) {
        $result = '<img src="' . $Image->getUrl() . '" />';
    } else {
        $result = QUI\Projects\Media\Utils::getImageHTML(
            $Image->getUrl(),
            $attributes,
            (bool)$host
        );
    }

    try {
        QUI::getEvents()->fireEvent('smartyImageEnd', [$smarty, &$result]);
    } catch (QUI\Exception $Exception) {
        QUI\System\Log::writeDebugException($Exception);
    }

    return smarty_plugin_image_assign($params, $result, $smarty);
}

/**
 * To assign the result to a variable
 *
 * @param array<string, mixed> $params
 * @param string $str
 * @param Smarty | Smarty_Internal_Template $smarty
 *
 * @return string
 */
function smarty_plugin_image_assign(array $params, string $str, Smarty | Smarty_Internal_Template $smarty): string
{
    if (!isset($params['assign'])) {
        return $str;
    }

    $smarty->assign($params['assign'], $str);

    return '';
}
