<?php

/**
 * This file contains QUI\TemplateTailwindCss\Controls\Children\ListPreview
 */

namespace QUI\TemplateTailwindCss\Controls\Children;

use QUI;
use QUI\Projects\Site\Utils;

/**
 * Class ListPreview
 *
 * @package quiqqer/template-tailwindcss
 */
class ListPreview extends QUI\Control
{
    /**
     * Path to list html file
     *
     * @var string
     */
    private string $listHTMLFile = '';

    /**
     * Path to list css file
     *
     * @var string
     */
    private string $listCSSFile = '';

    /**
     * Path to preview html file
     *
     * @var string
     */
    private string $previewHTMLFile = '';

    /**
     * Path to preview css file
     *
     * @var string
     */
    private string $previewCSSFile = '';

    /**
     * @var QUI\Interfaces\Template\EngineInterface|null
     */
    private ?QUI\Interfaces\Template\EngineInterface $Engine = null;

    /**
     * constructor
     *
     * @param array $attributes
     */
    public function __construct($attributes = [])
    {
        // default options
        $this->setAttributes([
            'nodeName' => 'section',
            'class' => 'qui-control-listPreview',
            'qui-class' => 'package/quiqqer/template-tailwindcss/bin/Controls/Children/SiteListPreview',
            'previewWidth' => 600,
            'openDirection' => 'left',
            'openAction' => 'hover',
            'order' => 'ASC',
            'targetBlank' => false,
            'entries' => [],
            // if true returns all sites of certain type
            'byType' => false,
            'where' => false,
            'itemtype' => 'http://schema.org/ItemList',
            'child-itemtype' => 'https://schema.org/ListItem',
            'child-itemprop' => 'itemListElement',
            // list layout / design
            'listTemplate' => 'default',
            // preview layout / design
            'previewTemplate' => 'default',
            'additionalText' => '',

            // overwrite
            // Custom list template (path to html file); overwrites "listTemplate"
            'customListTemplate' => false,
            // Custom list template css (path to css file); overwrites "listTemplate"
            'customListCss' => false,
            // Custom preview template (path to html file); overwrites "listTemplate"
            'customPreviewTemplate' => false,
            // Custom preview template css (path to css file); overwrites "listTemplate"
            'customPreviewCss' => false,

            // list of sites to display,
            'children' => false
        ]);

        parent::__construct($attributes);
    }

    /**
     * Return the inner body of the element
     * Can be overwritten
     *
     * @return String
     *
     * @throws QUI\Exception
     */
    public function getBody(): string
    {
        try {
            $this->Engine = QUI::getTemplateManager()->getEngine();
        } catch (QUI\Exception $Exception) {
            QUI\System\Log::writeException($Exception);

            return '';
        }

        $this->setJavaScriptControlOption('previewwidth', $this->getAttribute('previewWidth'));
        $this->setJavaScriptControlOption('opendirection', $this->getAttribute('openDirection'));
        $this->setJavaScriptControlOption('openaction', $this->getAttribute('openAction'));

        $entries = $this->getAttribute('entries');

        if (is_string($entries)) {
            $entries = json_decode($entries, true);
        }

        $additionalText = false;

        if ($this->getAttribute('additionalText') && \strlen($this->getAttribute('additionalText')) > 0) {
            $additionalText = $this->getAttribute('additionalText');
        }

        $targetBlank = false;

        if ($this->getAttribute('targetBlank')) {
            $targetBlank = $this->getAttribute('targetBlank');
        }

        $this->Engine->assign([
            'this' => $this,
            'Project' => $this->getProject(),
            'entries' => $entries,
            'targetBlank' => $targetBlank,
            'additionalText' => $additionalText
        ]);

        switch ($this->getAttribute('listTemplate')) {
            case 'default':
            case 'simpleOrdered':
            case 'simpleUnordered':
                $this->listCSSFile = \dirname(__FILE__) . '/ListPreview.List.' . $this->getAttribute(
                    'listTemplate'
                ) . '.css';
                $this->listHTMLFile = \dirname(__FILE__) . '/ListPreview.List.' . $this->getAttribute(
                    'listTemplate'
                ) . '.html';
                break;

            default:
                $this->listCSSFile = \dirname(__FILE__) . '/ListPreview.List.default.css';
                $this->listHTMLFile = \dirname(__FILE__) . '/ListPreview.List.default.html';
                break;
        }

        switch ($this->getAttribute('previewTemplate')) {
            case 'circle':
                $this->previewCSSFile = \dirname(__FILE__) . '/ListPreview.Preview.' . $this->getAttribute(
                    'previewTemplate'
                ) . '.css';
                $this->previewHTMLFile = \dirname(__FILE__) . '/ListPreview.Preview.' . $this->getAttribute(
                    'previewTemplate'
                ) . '.html';
                break;

            default:
                $this->previewCSSFile = \dirname(__FILE__) . '/ListPreview.Preview.circle.css';
                $this->previewHTMLFile = \dirname(__FILE__) . '/ListPreview.Preview.circle.html';
                break;
        }

        // load custom template (if set)
        $this->setCustomTemplates();

        $this->addCSSFiles([
            $this->getCSSFile(),
            $this->getPreviewCSSFile()
        ]);

        return $this->Engine->fetch($this->getHTMLFile());
    }

    /**
     * Get path to css file
     *
     * @return string
     */
    public function getCSSFile(): string
    {
        return $this->listCSSFile;
    }

    /**
     * Get path to html file
     *
     * @return string
     */
    public function getHTMLFile(): string
    {
        return $this->listHTMLFile;
    }

    /**
     * Get path to preview css file
     *
     * @return string
     */
    public function getPreviewCSSFile(): string
    {
        return $this->previewCSSFile;
    }

    /**
     * Get path to preview html file
     *
     * @return string
     */
    public function getPreviewHTMLFile(): string
    {
        return $this->previewHTMLFile;
    }

    /**
     * Get html template
     *
     * @return string
     */
    public function getPreview($params = []): string
    {
        $this->Engine->assign($params);

        return $this->Engine->fetch($this->getPreviewHTMLFile());
    }

    /**
     * Set custom files
     */
    protected function setCustomTemplates(): void
    {
        // list template
        if (
            $this->getAttribute('customListTemplate')
            && \file_exists($this->getAttribute('customListTemplate'))
        ) {
            if (
                $this->getAttribute('customListCss')
                && \file_exists($this->getAttribute('customListCss'))
            ) {
                $this->listCSSFile = $this->getAttribute('customListCss');
            }

            $this->listHTMLFile = $this->getAttribute('customListTemplate');
        }

        // preview template
        if (
            $this->getAttribute('customPreviewTemplate')
            && \file_exists($this->getAttribute('customPreviewTemplate'))
        ) {
            if (
                $this->getAttribute('customPreviewCss')
                && \file_exists($this->getAttribute('customPreviewCss'))
            ) {
                $this->listCSSFile = $this->getAttribute('customPreviewCss');
            }

            $this->listHTMLFile = $this->getAttribute('customPreviewTemplate');
        }
    }
}
