<?php

namespace QUI\Verification\Controls;

use DateMalformedStringException;
use Libellio\B2C\Core\User\Attribute\Verification\AddressVerificationHandler;
use QUI;
use QUI\Exception;
use QUI\Utils\Security\Orthos;
use QUI\Verification\Entity\AddressVerification as AddressVerificationEntity;
use QUI\Verification\Enum\VerificationStatus;
use QUI\Verification\Exception as VerificationException;
use QUI\Verification\Exception\CannotBuildVerificationException;
use QUI\Verification\Exception\InvalidVerificationCodeException;
use QUI\Verification\Exception\VerificationAlreadyVerifiedException;
use QUI\Verification\Exception\VerificationExpiredException;
use QUI\Verification\Interface\VerificationRepositoryInterface;
use QUI\Verification\Interface\VerifierInterface;
use QUI\Verification\VerificationRepository;
use QUI\Verification\Verifier;

use function is_null;

class AddressVerification extends QUI\Control
{
    private AddressVerificationEntity $addressVerification;

    /**
     * constructor.
     *
     * @param string $verificationUuid
     * @param VerificationRepositoryInterface|null $verificationRepository
     * @param VerifierInterface|null $verifier
     * @param mixed[] $attributes
     *
     * @throws Exception
     * @throws VerificationException
     * @throws CannotBuildVerificationException
     * @throws DateMalformedStringException
     * @throws \Doctrine\DBAL\Exception
     */
    public function __construct(
        string $verificationUuid,
        private ?VerificationRepositoryInterface $verificationRepository = null,
        private ?VerifierInterface $verifier = null,
        array $attributes = []
    ) {
        if (is_null($this->verificationRepository)) {
            $this->verificationRepository = new VerificationRepository();
        }

        if (is_null($this->verifier)) {
            $this->verifier = new Verifier();
        }

        $verification = $this->verificationRepository->findByUuid($verificationUuid);

        if (!($verification instanceof AddressVerificationEntity)) {
            throw new VerificationException("Verification not found: {$verificationUuid}");
        }

        $this->addressVerification = $verification;

        parent::__construct($attributes);
        $this->setAttributes($attributes);

        $this->addCSSFile(dirname(__FILE__) . '/AddressVerification.css');
        $this->addCSSClass('quiqqer-verification-AddressVerification');
    }

    /**
     * Return the control body
     *
     * @return string
     */
    public function getBody(): string
    {
        $Engine = QUI::getTemplateManager()->getEngine();
        $L = QUI::getLocale();
        $errorMsg = null;
        $showAddressAndCodeInput = true;
        $success = false;
        $onSuccessRedirectUrl = null;

        if ($this->addressVerification->status === VerificationStatus::VERIFIED) {
            $errorMsg = $L->get(
                'quiqqer/verification',
                'exception.controls.AddressVerification.error.already_verified'
            );
            $showAddressAndCodeInput = false;
        } elseif ($this->addressVerification->status === VerificationStatus::EXPIRED) {
            $errorMsg = $L->get(
                'quiqqer/verification',
                'exception.controls.AddressVerification.error.expired'
            );
            $showAddressAndCodeInput = false;
        } elseif (!empty($_POST['verificationCode'])) {
            $verificationCode = Orthos::clear($_POST['verificationCode']);

            try {
                // Just verify the code to catch exceptions
                // @phpstan-ignore-next-line
                $this->verifier->verifyAddressVerification($this->addressVerification, $verificationCode);

                /** @var AddressVerificationHandler $verificationHandler */
                $verificationHandler = $this->verificationRepository->getVerificationHandler(
                    $this->addressVerification
                );

                $onSuccessRedirectUrl = $verificationHandler->getOnSuccessRedirectUrl($this->addressVerification);
                $success = true;
            } catch (VerificationExpiredException $exception) {
                QUI\System\Log::writeDebugException($exception);
                $errorMsg = $L->get(
                    'quiqqer/verification',
                    'exception.controls.AddressVerification.error.expired'
                );
                $showAddressAndCodeInput = false;
            } catch (VerificationAlreadyVerifiedException $exception) {
                QUI\System\Log::writeDebugException($exception);
                $errorMsg = $L->get(
                    'quiqqer/verification',
                    'exception.controls.AddressVerification.error.already_verified'
                );
                $showAddressAndCodeInput = false;
            } catch (InvalidVerificationCodeException $exception) {
                QUI\System\Log::writeDebugException($exception);
                $errorMsg = $L->get(
                    'quiqqer/verification',
                    'exception.controls.AddressVerification.error.invalid_code'
                );
            } catch (\Exception | \Throwable $exception) {
                if ($exception instanceof Exception) {
                    QUI\System\Log::writeException($exception);
                }
                $errorMsg = $L->get(
                    'quiqqer/verification',
                    'exception.controls.AddressVerification.error.general'
                );
            }
        }

        $Engine->assign([
            'success' => $success,
            'onSuccessRedirectUrl' => $onSuccessRedirectUrl,
            'showAddressAndCodeInput' => $showAddressAndCodeInput,
            'errorMsg' => $errorMsg,
            'address' => $this->addressVerification->address,
            'verificationCode' => $this->addressVerification->verificationCode
        ]);

        return $Engine->fetch(dirname(__FILE__) . '/AddressVerification.html');
    }
}
