<?php

namespace QUI\Verification;

use QUI;
use QUI\Verification\Enum\VerificationStatus;

use function time;

/**
 * Class Cron
 *
 * Cron class for quiqqer/verification
 */
class Cron
{
    /**
     * Delete all verified Verification entries from database
     * that have been verified a certain amount of days before now
     *
     * @return void
     */
    public static function deleteVerified(): void
    {
        $repository = new VerificationRepository();

        try {
            $deletionCandidates = $repository->findAllByStatus(VerificationStatus::VERIFIED);
        } catch (\Throwable $exception) {
            if ($exception instanceof \Exception) {
                QUI\System\Log::writeException($exception);
            }
            return;
        }

        $deleteVerifications = [];

        try {
            $Conf = QUI::getPackage('quiqqer/verification')->getConfig();
        } catch (\Exception $exception) {
            QUI\System\Log::writeException($exception);
            return;
        }

        if (is_null($Conf)) {
            QUI\System\Log::addError("quiqqer/verification cron :: Cannot load Config for quiqqer/verification.");
            return;
        }

        $verifiedTtl = (int)$Conf->get('settings', 'keepVerifiedDuration'); // days
        $verifiedTtl *= 24 * 60 * 60; // seconds
        $now = time();

        foreach ($deletionCandidates as $verification) {
            $verifiedTime = $verification->updateDate->getTimestamp();
            $aliveTime = $now - $verifiedTime;

            if ($aliveTime > $verifiedTtl) {
                $deleteVerifications[] = $verification;
            }
        }

        foreach ($deleteVerifications as $verification) {
            try {
                $repository->delete($verification);
            } catch (\Throwable $exception) {
                if ($exception instanceof \Exception) {
                    QUI\System\Log::writeException($exception);
                }
            }
        }
    }

    /**
     * Delete all unverified Verification entries from database
     * that have exceeded their "valid until" date
     *
     * @return void
     */
    public static function deleteUnverified(): void
    {
        $repository = new VerificationRepository();

        try {
            $unverified = $repository->findAllUnverifiedAndInvalid();
        } catch (\Throwable $exception) {
            if ($exception instanceof \Exception) {
                QUI\System\Log::writeException($exception);
            }
            return;
        }


        foreach ($unverified as $verification) {
            try {
                QUI::getEvents()->fireEvent(
                    'quiqqerVerificationDeleteUnverified',
                    [
                        $verification
                    ]
                );
            } catch (\Exception $exception) {
                QUI\System\Log::writeDebugException($exception);
                continue;
            }

            try {
                $repository->delete($verification);
            } catch (\Throwable $exception) {
                if ($exception instanceof \Exception) {
                    QUI\System\Log::writeException($exception);
                }
            }
        }
    }
}
