<?php

namespace QUI\Verification\Interface;

use QUI\PhoneApi\Entity\PhoneNumber;
use QUI\Verification\Entity\Address;
use QUI\Verification\Entity\AddressVerification;
use QUI\Verification\Entity\LinkVerification;
use QUI\Verification\Entity\PhoneNumberVerification;
use QUI\Verification\Enum\PhoneNumberVerificationStrategy;

interface VerificationFactoryInterface
{
    /**
     * Start a verification process via link.
     *
     * @param string $identifier - A custom unique identifier that is used to determine identical verifications;
     * e.g. an email address for an email address verification process.
     * @param LinkVerificationHandlerInterface $verificationHandler
     * @param array<string|int,mixed> $customData (optional) - Custom data that is stored with the Verification
     * @param bool $overwriteExisting (optional) - Overwrite Verification with identical
     * identifier and source class [default: false]
     * @return LinkVerification
     */
    public function createLinkVerification(
        string $identifier,
        LinkVerificationHandlerInterface $verificationHandler,
        array $customData = [],
        bool $overwriteExisting = false
    ): LinkVerification;

    /**
     * Create verification for verifying a phone number via voice call.
     *
     * @param string $identifier - A custom unique identifier that is used to determine identical verifications;
     * e.g. an email address for an email address verification process.
     * @param PhoneNumber $phoneNumber
     * @param PhoneNumberVerificationStrategy $strategy
     * @param PhoneNumberVerificationHandlerInterface $verificationHandler
     * @param array<string|int,mixed> $customData (optional) - Custom data that is stored with the Verification
     * @param string|null $verificationCode (optional) - The verification code used. If NULL, generate random code
     * @param bool $overwriteExisting (optional) - Overwrite Verification with identical
     * identifier and source class [default: false]
     * @return PhoneNumberVerification
     */
    public function createPhoneNumberVerification(
        string $identifier,
        PhoneNumber $phoneNumber,
        PhoneNumberVerificationStrategy $strategy,
        PhoneNumberVerificationHandlerInterface $verificationHandler,
        array $customData = [],
        ?string $verificationCode = null,
        bool $overwriteExisting = false
    ): PhoneNumberVerification;

    /**
     * Create verification for verifying an address via code that is sent via letter.
     *
     * @param string $identifier - A custom unique identifier that is used to determine identical verifications;
     * e.g. an email address for an email address verification process.
     * @param Address $address
     * @param AddressVerificationHandlerInterface $verificationHandler
     * @param array<string|int,mixed> $customData (optional) - Custom data that is stored with the Verification
     * @param string|null $verificationCode (optional) - The verification code used. If NULL, generate random code
     * @param bool $overwriteExisting (optional) - Overwrite Verification with identical
     * identifier and source class [default: false]
     * @return AddressVerification
     */
    public function createAddressVerification(
        string $identifier,
        Address $address,
        AddressVerificationHandlerInterface $verificationHandler,
        array $customData = [],
        ?string $verificationCode = null,
        bool $overwriteExisting = false
    ): AddressVerification;
}
