<?php

namespace QUI\Verification;

use Composer\Semver\Semver;
use QUI;

use function is_null;

class Utils
{
    /**
     * Verifier site type
     */
    const SITE_TYPE_VERIFIER = 'quiqqer/verification:types/verifier';
    const SITE_TYPE_VERIFY_ADDRESS = 'quiqqer/verification:types/verifyAddress';

    /**
     * Get formatted timestamp for a given UNIX timestamp
     *
     * @param int|null $time (optional) - if omitted use time()
     * @return string
     */
    public static function getFormattedTimestamp(?int $time = null): string
    {
        if (is_null($time)) {
            $time = time();
        }

        return date('Y-m-d H:i:s', $time);
    }

    /**
     * Get verifier Site
     *
     * @return QUI\Interfaces\Projects\Site
     * @throws QUI\Verification\Exception
     * @throws QUI\Database\Exception|QUI\Exception
     */
    public static function getVerifierSite(): QUI\Interfaces\Projects\Site
    {
        $Project = QUI::getRewrite()->getProject();

        if (is_null($Project)) {
            throw new Exception("Cannot load QUIQQER project to fetch verifier site.");
        }

        $siteIds = $Project->getSitesIds([
            'where' => [
                'type' => self::SITE_TYPE_VERIFIER
            ]
        ]);

        if (empty($siteIds)) {
            throw new QUI\Verification\Exception([
                'quiqqer/verification',
                'exception.verifier.site.does.not.exist'
            ]);
        }

        return $Project->get($siteIds[0]['id']);
    }

    /**
     * Get verifier Site for address verification.
     *
     * @return QUI\Interfaces\Projects\Site
     * @throws QUI\Verification\Exception
     * @throws QUI\Database\Exception|QUI\Exception
     */
    public static function getAddressVerifierSite(): QUI\Interfaces\Projects\Site
    {
        $Project = QUI::getRewrite()->getProject();

        if (is_null($Project)) {
            throw new Exception("Cannot load QUIQQER project to fetch verifier site.");
        }

        $siteIds = $Project->getSitesIds([
            'where' => [
                'type' => self::SITE_TYPE_VERIFY_ADDRESS
            ]
        ]);

        if (empty($siteIds)) {
            throw new QUI\Verification\Exception([
                'quiqqer/verification',
                'exception.verifier.site.does.not.exist'
            ]);
        }

        return $Project->get($siteIds[0]['id']);
    }

    /**
     * Check if a package is installed in a required version.
     *
     * @param string $package
     * @param string $requiredVersion
     * @return bool
     */
    public static function isPackageInstalledInRequiredVersion(string $package, string $requiredVersion): bool
    {
        if (!QUI::getPackageManager()->isInstalled($package)) {
            return false;
        }

        try {
            $package = QUI::getPackage($package);
            return Semver::satisfies($package->getVersion(), $requiredVersion);
        } catch (\Exception $exception) {
            QUI\System\Log::writeDebugException($exception);
            return false;
        }
    }

    /**
     * Create a hash from a given identifier.
     *
     * @param string $identifier
     * @return string
     */
    public static function hashIdentifier(string $identifier): string
    {
        return hash('sha256', $identifier);
    }
}
