<?php

namespace QUI\Verification;

use DateMalformedStringException;
use DateTimeImmutable;
use QUI;
use QUI\Database\Exception;
use QUI\PhoneApi\Entity\PhoneNumber;
use QUI\Verification\Entity\AbstractVerification;
use QUI\Verification\Entity\Address;
use QUI\Verification\Entity\AddressVerification;
use QUI\Verification\Entity\LinkVerification;
use QUI\Verification\Entity\PhoneNumberVerification;
use QUI\Verification\Enum\PhoneNumberVerificationStrategy;
use QUI\Verification\Enum\VerificationStatus;
use QUI\Verification\Exception\VerificationCooldownActiveException;
use QUI\Verification\Interface\AddressVerificationHandlerInterface;
use QUI\Verification\Interface\LinkVerificationHandlerInterface;
use QUI\Verification\Interface\PhoneNumberVerificationHandlerInterface;
use QUI\Verification\Interface\VerificationCodeFactoryInterface;
use QUI\Verification\Interface\VerificationFactoryInterface;
use QUI\Verification\Interface\VerificationHandlerInterface;
use QUI\Verification\Interface\VerificationRepositoryInterface;

use function date_create;
use function date_interval_create_from_date_string;
use function is_null;
use function strtotime;

class VerificationFactory implements VerificationFactoryInterface
{
    private VerificationRepositoryInterface $verificationRepository;
    private VerificationCodeFactoryInterface $verificationCodeFactory;

    /**
     * @param VerificationRepositoryInterface|null $verificationRepository
     * @param VerificationCodeFactoryInterface|null $verificationCodeFactory
     */
    public function __construct(
        ?VerificationRepositoryInterface $verificationRepository = null,
        ?VerificationCodeFactoryInterface $verificationCodeFactory = null
    ) {
        $this->verificationRepository = $verificationRepository ?? new VerificationRepository();
        $this->verificationCodeFactory = $verificationCodeFactory ?? new VerificationCodeFactory();
    }

    /**
     * Start a verification process
     *
     * @param string $identifier - A custom unique identifier that is used to determine identical verifications;
     * e.g. an email address for an email address verification process.
     * @param LinkVerificationHandlerInterface $verificationHandler
     * @param array<string|int,mixed> $customData (optional) - Custom data that is stored with the Verification
     * @param bool $overwriteExisting (optional) - Overwrite Verification with identical
     * identifier and source class [default: false]
     * @return LinkVerification
     *
     * @throws QUI\Verification\Exception
     * @throws QUI\Exception
     * @throws DateMalformedStringException
     */
    public function createLinkVerification(
        string $identifier,
        LinkVerificationHandlerInterface $verificationHandler,
        array $customData = [],
        bool $overwriteExisting = false
    ): LinkVerification {
        $existingVerification = $this->verificationRepository->findByIdentifier($identifier);

        if (!is_null($existingVerification)) {
            if ($overwriteExisting !== true) {
                throw new QUI\Verification\Exception([
                    'quiqqer/verification',
                    'exception.verifier.verification.already.exists',
                    [
                        'identifier' => $existingVerification->identifier
                    ]
                ]);
            }

            $this->verificationRepository->delete($existingVerification);
        }

        $uuid = QUI\Utils\Uuid::get();
        $hash = $this->verificationCodeFactory->createRandomHashCode();
        $verifierSite = Utils::getVerifierSite();
        $url = '';

        if (method_exists($verifierSite, 'getUrlRewrittenWithHost')) {
            $url = $verifierSite->getUrlRewrittenWithHost([], [
                'verificationId' => $uuid,
                'hash' => $hash
            ]);
        }

        $linkVerification = new LinkVerification(
            $uuid,
            $identifier,
            $hash,
            new DateTimeImmutable(),
            new DateTimeImmutable(),
            0,
            $url,
            VerificationStatus::PENDING,
            $customData
        );

        $this->setValidUntilDateToVerification($linkVerification, $verificationHandler);
        $this->verificationRepository->insert($linkVerification, $verificationHandler);

        return $linkVerification;
    }

    /**
     * Create verification for verifying a phone number via voice call.
     *
     * @param string $identifier - A custom unique identifier that is used to determine identical verifications;
     * e.g. an email address for an email address verification process.
     * @param PhoneNumber $phoneNumber
     * @param PhoneNumberVerificationStrategy $strategy
     * @param PhoneNumberVerificationHandlerInterface $verificationHandler
     * @param array<string|int,mixed> $customData (optional) - Custom data that is stored with the Verification
     * @param string|null $verificationCode (optional) - The verification code used. If NULL, generate random code
     * @param bool $overwriteExisting (optional) - Overwrite Verification with identical
     * identifier and source class [default: false]
     * @return PhoneNumberVerification
     *
     * @throws DateMalformedStringException
     * @throws QUI\Exception
     * @throws Exception
     */
    public function createPhoneNumberVerification(
        string $identifier,
        PhoneNumber $phoneNumber,
        PhoneNumberVerificationStrategy $strategy,
        PhoneNumberVerificationHandlerInterface $verificationHandler,
        array $customData = [],
        ?string $verificationCode = null,
        bool $overwriteExisting = false
    ): PhoneNumberVerification {
        $existingVerification = $this->verificationRepository->findByIdentifier($identifier);

        if (!is_null($existingVerification)) {
            if ($overwriteExisting !== true) {
                throw new QUI\Verification\Exception([
                    'quiqqer/verification',
                    'exception.verifier.verification.already.exists',
                    [
                        'identifier' => $existingVerification->identifier
                    ]
                ]);
            }

            $this->checkCooldown($identifier, (int)Settings::get('cooldownPhoneVerification'));
            $this->verificationRepository->delete($existingVerification);
        }

        $uuid = QUI\Utils\Uuid::get();

        if (is_null($verificationCode)) {
            $verificationCode = $this->verificationCodeFactory->createRandomDigitsCode();
        }

        $phoneNumberVerification = new PhoneNumberVerification(
            $uuid,
            $identifier,
            $verificationCode,
            new DateTimeImmutable(),
            new DateTimeImmutable(),
            0,
            $phoneNumber,
            $strategy,
            VerificationStatus::PENDING,
            $customData
        );

        $this->setValidUntilDateToVerification($phoneNumberVerification, $verificationHandler);
        $this->verificationRepository->insert($phoneNumberVerification, $verificationHandler);

        return $phoneNumberVerification;
    }

    /**
     * Create verification for verifying an address via code that is sent via letter.
     *
     * @param string $identifier - A custom unique identifier that is used to determine identical verifications;
     * e.g. an email address for an email address verification process.
     * @param Address $address
     * @param AddressVerificationHandlerInterface $verificationHandler
     * @param array<string|int,mixed> $customData (optional) - Custom data that is stored with the Verification
     * @param string|null $verificationCode (optional) - The verification code used. If NULL, generate random code
     * @param bool $overwriteExisting (optional) - Overwrite Verification with identical
     * identifier and source class [default: false]
     * @return AddressVerification
     *
     * @throws DateMalformedStringException
     * @throws QUI\Exception
     * @throws Exception
     */
    public function createAddressVerification(
        string $identifier,
        Address $address,
        AddressVerificationHandlerInterface $verificationHandler,
        array $customData = [],
        ?string $verificationCode = null,
        bool $overwriteExisting = false
    ): AddressVerification {
        $existingVerification = $this->verificationRepository->findByIdentifier($identifier);

        if (!is_null($existingVerification)) {
            if ($overwriteExisting !== true) {
                throw new QUI\Verification\Exception([
                    'quiqqer/verification',
                    'exception.verifier.verification.already.exists',
                    [
                        'identifier' => $existingVerification->identifier
                    ]
                ]);
            }

            $this->checkCooldown($identifier, (int)Settings::get('cooldownAddressVerification'));
            $this->verificationRepository->delete($existingVerification);
        }

        $uuid = QUI\Utils\Uuid::get();

        if (is_null($verificationCode)) {
            $verificationCode = $this->verificationCodeFactory->createRandomDigitsCode();
        }

        $verifierSite = Utils::getAddressVerifierSite();
        $url = '';

        if (method_exists($verifierSite, 'getUrlRewrittenWithHost')) {
            $url = $verifierSite->getUrlRewrittenWithHost([], [
                'id' => $uuid,
            ]);
        }

        $addressVerification = new AddressVerification(
            $uuid,
            $identifier,
            $verificationCode,
            new DateTimeImmutable(),
            new DateTimeImmutable(),
            0,
            $address,
            $url,
            VerificationStatus::PENDING,
            $customData
        );

        $this->setValidUntilDateToVerification($addressVerification, $verificationHandler);
        $this->verificationRepository->insert($addressVerification, $verificationHandler);

        return $addressVerification;
    }

    /**
     * @param AbstractVerification $verification
     * @param VerificationHandlerInterface $handler
     * @return void
     *
     * @throws DateMalformedStringException
     * @throws QUI\Exception
     */
    private function setValidUntilDateToVerification(
        AbstractVerification $verification,
        VerificationHandlerInterface $handler
    ): void {
        $validDuration = $handler->getValidDuration($verification);

        // fallback
        if (empty($validDuration)) {
            $Conf = QUI::getPackage('quiqqer/verification')->getConfig();

            if (is_null($Conf)) {
                $validDuration = 4320; // fallback: 3 days

                QUI\System\Log::addError(
                    "quiqqer/verification :: VerificationFactory -> cannot load Config of quiqqer/verification"
                    . " for determining default verification validity duration. Falling back to 3 days hardcoded."
                );
            } else {
                $validDuration = $Conf->get('settings', 'validDuration');
            }
        }

        // calculate duration
        $end = strtotime(
            Utils::getFormattedTimestamp() . ' +' . $validDuration . ' minute'
        );

        $verification->validUntilDate = new DateTimeImmutable('@' . $end);
    }

    /**
     * Check if a verification with a specific identifier can be recreated.
     *
     * @param string $identifier
     * @param int $cooldownMinutes
     * @return void
     * @throws VerificationCooldownActiveException
     */
    private function checkCooldown(string $identifier, int $cooldownMinutes): void
    {
        $existingVerification = $this->verificationRepository->findByIdentifier($identifier);
        $now = date_create();
        $cooldownExpirationDate = clone $existingVerification->createDate;
        $cooldownMinutesInterval = date_interval_create_from_date_string($cooldownMinutes . ' minutes');

        if ($cooldownMinutesInterval === false) {
            QUI\System\Log::addWarning(
                "Cannot calculate verification cooldown minutes from value $cooldownMinutes."
            );
            return;
        }

        $cooldownExpirationDate = $cooldownExpirationDate->add($cooldownMinutesInterval);

        if ($now >= $cooldownExpirationDate) {
            return;
        }

        throw new VerificationCooldownActiveException(
            $cooldownExpirationDate,
            "Verification with identifier '$identifier' is still on cooldown. Can be recreated at "
            . $cooldownExpirationDate->format('Y-m-d H:i:s') . " at the earliest."
        );
    }
}
