<?php

namespace QUI\Verification;

use DateMalformedStringException;
use Doctrine\DBAL\Connection;
use QUI;
use QUI\Exception;
use QUI\Security\Encryption;
use QUI\Verification\Entity\AbstractVerification;
use QUI\Verification\Entity\LinkVerification;
use QUI\Verification\Entity\PhoneNumberVerification;
use QUI\Verification\Enum\VerificationStatus;
use QUI\Verification\Exception\CannotBuildVerificationException;
use QUI\Verification\Interface\VerificationHandlerInterface;
use QUI\Verification\Interface\VerificationRepositoryInterface;
use QUI\Verification\Entity\AddressVerification;

use function date;
use function is_null;
use function json_decode;
use function json_encode;

class VerificationRepository implements VerificationRepositoryInterface
{
    const TBL_VERIFICATION_PROCESSES = 'quiqqer_verification_processes';

    private Connection $databaseConnection;

    public function __construct(?Connection $databaseConnection = null)
    {
        if ($databaseConnection) {
            $this->databaseConnection = $databaseConnection;
        }

        if (is_null($databaseConnection)) {
            $this->databaseConnection = QUI::getDataBaseConnection();
        }
    }

    /**
     * @param string $uuid
     * @return AbstractVerification|null
     *
     * @throws \DateMalformedStringException
     * @throws \Doctrine\DBAL\Exception
     * @throws Exception
     * @throws CannotBuildVerificationException
     */
    public function findByUuid(string $uuid): ?AbstractVerification
    {
        $result = $this->databaseConnection
            ->createQueryBuilder()
            ->select('*')
            ->from($this->tableVerificationProcesses())
            ->where('uuid = :uuid')
            ->setParameter('uuid', $uuid)
            ->setMaxResults(1)
            ->fetchAssociative();

        return $result ? $this->parseDbRowToVerification($result) : null;
    }

    /**
     * @param string $identifier
     * @return AbstractVerification|null
     *
     * @throws Exception
     * @throws DateMalformedStringException
     * @throws \Doctrine\DBAL\Exception
     * @throws CannotBuildVerificationException
     */
    public function findByIdentifier(
        string $identifier
    ): ?AbstractVerification {
        $result = $this->databaseConnection
            ->createQueryBuilder()
            ->select('*')
            ->from($this->tableVerificationProcesses())
            ->where('identifier = :identifier')
            ->setParameter('identifier', $identifier)
            ->setMaxResults(1)
            ->fetchAssociative();

        return $result ? $this->parseDbRowToVerification($result) : null;
    }

    /**
     * @param AbstractVerification $verification
     * @param VerificationHandlerInterface $verificationHandler
     * @return void
     *
     * @throws DateMalformedStringException
     * @throws Exception
     * @throws \Doctrine\DBAL\Exception
     * @throws \Throwable
     */
    public function insert(AbstractVerification $verification, VerificationHandlerInterface $verificationHandler): void
    {
        if (!is_null($this->findByUuid($verification->uuid))) {
            throw new \InvalidArgumentException('Verification already exists');
        }

        $customDataEncoded = null;

        if (!empty($verification->getCustomData())) {
            $customDataEncoded = json_encode($verification->getCustomData());

            if ($customDataEncoded === false) {
                throw new Exception("Could not encode verification custom data");
            }
        }

        $this->databaseConnection->insert(
            $this->tableVerificationProcesses(),
            [
                'uuid' => $verification->uuid,
                'identifier' => $verification->identifier,
                'verificationClass' => $verification::class,
                'customData' => $customDataEncoded ?
                    Encryption::encrypt($customDataEncoded) :
                    null,
                'verificationCode' => Encryption::encrypt($verification->verificationCode),
                'createDate' => $verification->createDate->format('Y-m-d H:i:s'),
                'updateDate' => $verification->updateDate->format('Y-m-d H:i:s'),
                'validUntilDate' => $verification->validUntilDate?->format('Y-m-d H:i:s'),
                'verificationHandler' => $verificationHandler::class,
                'status' => $verification->status->value
            ]
        );
    }

    /**
     * @param AbstractVerification $verification
     * @return void
     * @throws \Throwable
     */
    public function update(AbstractVerification $verification): void
    {
        $this->databaseConnection->update(
            $this->tableVerificationProcesses(),
            [
                'status' => $verification->status->value,
                'updateDate' => date('Y-m-d H:i:s'),
                'tries' => $verification->tries
            ],
            [
                'uuid' => $verification->uuid
            ]
        );
    }

    /**
     * @param AbstractVerification $verification
     * @return void
     * @throws \Throwable
     */
    public function delete(AbstractVerification $verification): void
    {
        $this->databaseConnection->delete(
            $this->tableVerificationProcesses(),
            [
                'uuid' => $verification->uuid
            ]
        );
    }

    /**
     * Get instance of the verification handler for a verification.
     *
     * @param AbstractVerification $verification
     * @return VerificationHandlerInterface
     * @throws \Doctrine\DBAL\Exception
     * @throws Exception
     */
    public function getVerificationHandler(AbstractVerification $verification): VerificationHandlerInterface
    {
        $result = $this->databaseConnection
            ->createQueryBuilder()
            ->select('verificationHandler')
            ->from($this->tableVerificationProcesses())
            ->where('uuid = :uuid')
            ->setParameter('uuid', $verification->uuid)
            ->setMaxResults(1)
            ->fetchAssociative();

        if (empty($result['verificationHandler'])) {
            throw new Exception("Cannot load verification handler -> Missing key 'verificationHandler' in result row.");
        }

        $class = $result['verificationHandler'];
        /** @var VerificationHandlerInterface $handler */
        $handler = new $class();
        return $handler;
    }

    /**
     * Finds all verifications that are unverified (pending) and exceeded their validity date.
     *
     * @return AbstractVerification[]
     * @throws Exception
     * @throws QUI\Exception
     * @throws \DateMalformedStringException
     * @throws \Doctrine\DBAL\Exception
     * @throws CannotBuildVerificationException
     */
    public function findAllUnverifiedAndInvalid(): array
    {
        $verifications = [];

        $result = $this->databaseConnection
            ->createQueryBuilder()
            ->select('*')
            ->from($this->tableVerificationProcesses())
            ->where('status = :pendingStatus')
            ->andWhere('validUntilDate <= :validUntilDate')
            ->andWhere('validUntilDate IS NOT NULL')
            ->setParameter('pendingStatus', VerificationStatus::PENDING->value)
            ->setParameter('validUntilDate', date('Y-m-d H:i:s'))
            ->fetchAllAssociative();

        foreach ($result as $row) {
            $verifications[] = $this->parseDbRowToVerification($row);
        }

        return $verifications;
    }

    /**
     * @param VerificationStatus $status
     * @return AbstractVerification[]
     * @throws Exception
     * @throws QUI\Exception
     * @throws CannotBuildVerificationException
     * @throws \Doctrine\DBAL\Exception
     */
    public function findAllByStatus(VerificationStatus $status): array
    {
        $verifications = [];

        $result = $this->databaseConnection
            ->createQueryBuilder()
            ->select('*')
            ->from($this->tableVerificationProcesses())
            ->where('status = :status')
            ->setParameter('status', $status->value)
            ->fetchAllAssociative();

        foreach ($result as $row) {
            $verifications[] = $this->parseDbRowToVerification($row);
        }

        return $verifications;
    }

    /**
     * @param array<string,mixed> $row
     * @return AbstractVerification
     *
     * @throws Exception
     * @throws CannotBuildVerificationException
     */
    private function parseDbRowToVerification(array $row): AbstractVerification
    {
        $row['verificationCode'] = Encryption::decrypt($row['verificationCode']);
        $row['customData'] = !empty($row['customData']) ?
            json_decode(Encryption::decrypt($row['customData']), true) :
            [];

        switch ($row['verificationClass']) {
            case LinkVerification::class:
                try {
                    return LinkVerification::fromArray($row);
                } catch (\Exception $exception) {
                    QUI\System\Log::writeException($exception);
                    throw new CannotBuildVerificationException(
                        "Cannot build " . LinkVerification::class . " from database row #{$row['id']}"
                    );
                }

            case PhoneNumberVerification::class:
                try {
                    return PhoneNumberVerification::fromArray($row);
                } catch (\Exception $exception) {
                    QUI\System\Log::writeException($exception);
                    throw new CannotBuildVerificationException(
                        "Cannot build " . PhoneNumberVerification::class . " from database row #{$row['id']}"
                    );
                }

            case AddressVerification::class:
                try {
                    return AddressVerification::fromArray($row);
                } catch (\Exception $exception) {
                    QUI\System\Log::writeException($exception);
                    throw new CannotBuildVerificationException(
                        "Cannot build " . AddressVerification::class . " from database row #{$row['id']}"
                    );
                }

            default:
                throw new Exception("Invalid verification class: {$row['verificationClass']}");
        }
    }

    private function tableVerificationProcesses(): string
    {
        return QUI::getDBTableName(self::TBL_VERIFICATION_PROCESSES);
    }
}
