<?php

namespace QUI\Verification;

use QUI;
use QUI\Exception;
use QUI\Verification\Entity\AbstractVerification;
use QUI\Verification\Entity\AddressVerification;
use QUI\Verification\Entity\PhoneNumberVerification;
use QUI\Verification\Enum\VerificationErrorReason;
use QUI\Verification\Enum\VerificationStatus;
use QUI\Verification\Exception\InvalidVerificationCodeException;
use QUI\Verification\Exception\VerificationAlreadyVerifiedException;
use QUI\Verification\Exception\VerificationExpiredException;
use QUI\Verification\Interface\AddressVerificationHandlerInterface;
use QUI\Verification\Interface\PhoneNumberVerificationHandlerInterface;
use QUI\Verification\Interface\VerificationRepositoryInterface;
use QUI\Verification\Interface\VerifierInterface;

use function is_null;

class Verifier implements VerifierInterface
{
    /**
     * @param VerificationRepositoryInterface|null $verificationRepository
     */
    public function __construct(protected ?VerificationRepositoryInterface $verificationRepository = null)
    {
        if (is_null($this->verificationRepository)) {
            $this->verificationRepository = new VerificationRepository();
        }
    }

    /**
     * @param AbstractVerification $verification
     * @param string $verificationCode
     * @return void
     *
     * @throws InvalidVerificationCodeException
     * @throws VerificationAlreadyVerifiedException
     * @throws VerificationExpiredException
     * @throws QUI\Exception
     */
    public function verifyVerificationCode(AbstractVerification $verification, string $verificationCode): void
    {
        if ($verification->status === VerificationStatus::VERIFIED) {
            throw new VerificationAlreadyVerifiedException(
                "Verification {$verification->uuid} is already verified."
            );
        }

        if ($verification->status === VerificationStatus::EXPIRED) {
            throw new VerificationExpiredException(
                "Verification {$verification->uuid} is expired."
            );
        }

        // If expired but verification instance does not reflect that -> update state
        if ($verification->isValid() === false) {
            $verification->status = VerificationStatus::EXPIRED;
            $this->verificationRepository->update($verification); // @phpstan-ignore-line

            throw new VerificationExpiredException(
                "Verification {$verification->uuid} is expired."
            );
        }

        // Check max tries -> if exceeded, set status to expired
        $maxTries = (int)Settings::get('maxTries');

        if ($verification->tries >= $maxTries) {
            $verification->status = VerificationStatus::EXPIRED;
            $this->verificationRepository->update($verification); // @phpstan-ignore-line

            throw new VerificationExpiredException(
                "Verification {$verification->uuid} is expired."
            );
        }

        if ($verification->verificationCode !== $verificationCode) {
            // Increase tries
            $verification->tries += 1;
            $this->verificationRepository->update($verification); // @phpstan-ignore-line

            throw new InvalidVerificationCodeException(
                "Verification code {$verificationCode} is invalid for verification {$verification->uuid}."
            );
        }

        $verification->status = VerificationStatus::VERIFIED;
        $this->verificationRepository->update($verification);  // @phpstan-ignore-line
    }

    /**
     * @param PhoneNumberVerification $verification
     * @param string $verificationCode
     * @return void
     *
     * @throws Exception
     * @throws \Doctrine\DBAL\Exception
     * @throws InvalidVerificationCodeException
     * @throws VerificationAlreadyVerifiedException
     * @throws VerificationExpiredException
     */
    public function verifyPhoneNumberVerification(PhoneNumberVerification $verification, string $verificationCode): void
    {
        /** @var PhoneNumberVerificationHandlerInterface $verificationHandler */
        $verificationHandler = $this->verificationRepository->getVerificationHandler($verification); // @phpstan-ignore-line

        try {
            $this->verifyVerificationCode($verification, $verificationCode);
            $verificationHandler->onSuccess($verification);
        } catch (VerificationAlreadyVerifiedException $exception) {
            QUI\System\Log::writeDebugException($exception);
            $verificationHandler->onError($verification, VerificationErrorReason::ALREADY_VERIFIED);
            throw $exception;
        } catch (VerificationExpiredException $exception) {
            QUI\System\Log::writeDebugException($exception);
            $verificationHandler->onError($verification, VerificationErrorReason::EXPIRED);
            throw $exception;
        } catch (InvalidVerificationCodeException $exception) {
            QUI\System\Log::writeDebugException($exception);
            $verificationHandler->onError($verification, VerificationErrorReason::INVALID_CODE);
            throw $exception;
        }
    }

    /**
     * @param AddressVerification $verification
     * @param string $verificationCode
     * @return void
     *
     * @throws Exception
     * @throws \Doctrine\DBAL\Exception
     * @throws InvalidVerificationCodeException
     * @throws VerificationAlreadyVerifiedException
     * @throws VerificationExpiredException
     */
    public function verifyAddressVerification(AddressVerification $verification, string $verificationCode): void
    {
        /** @var AddressVerificationHandlerInterface $verificationHandler */
        // @phpstan-ignore-next-line
        $verificationHandler = $this->verificationRepository->getVerificationHandler($verification);

        try {
            $this->verifyVerificationCode($verification, $verificationCode);
            $verificationHandler->onSuccess($verification);
        } catch (VerificationAlreadyVerifiedException $exception) {
            QUI\System\Log::writeDebugException($exception);
            $verificationHandler->onError($verification, VerificationErrorReason::ALREADY_VERIFIED);
            throw $exception;
        } catch (VerificationExpiredException $exception) {
            QUI\System\Log::writeDebugException($exception);
            $verificationHandler->onError($verification, VerificationErrorReason::EXPIRED);
            throw $exception;
        } catch (InvalidVerificationCodeException $exception) {
            QUI\System\Log::writeDebugException($exception);
            $verificationHandler->onError($verification, VerificationErrorReason::INVALID_CODE);
            throw $exception;
        }
    }
}
