<?php

/**
 * @var QUI\Projects\Project $Project
 * @var QUI\Projects\Site $Site
 * @var QUI\Interfaces\Template\EngineInterface $Engine
 * @var QUI\Template $Template
 **/

use QUI\Utils\Security\Orthos;
use QUI\Verification\Enum\VerificationErrorReason;
use QUI\Verification\Exception\InvalidVerificationCodeException;
use QUI\Verification\Exception\VerificationAlreadyVerifiedException;
use QUI\Verification\Exception\VerificationExpiredException;
use QUI\Verification\VerificationRepository;
use QUI\Verification\Verifier;
use QUI\Verification\Entity\LinkVerification;

function redirect(string $target): never
{
    header('Location: ' . $target);
    exit;
}

if (empty($_REQUEST['hash']) || empty($_REQUEST['verificationId'])) {
    $Engine->assign([
        'msg' => QUI::getLocale()->get('quiqqer/verification', 'message.types.verifier.error.general'),
        'success' => false
    ]);

    return;
}

$success = false;
$msg = false;
$errorReason = VerificationErrorReason::INVALID_REQUEST;
$verification = null;
$repository = new VerificationRepository();

try {
    $verification = $repository->findByUuid(Orthos::clear($_REQUEST['verificationId']));

    if (!($verification instanceof LinkVerification)) {
        $verification = null;
    }
} catch (\Throwable $Exception) {
    if ($Exception instanceof Exception) {
        QUI\System\Log::writeDebugException($Exception);
    }
}

if (is_null($verification)) {
    $Engine->assign([
        'msg' => QUI::getLocale()->get('quiqqer/verification', 'message.types.verifier.error.not_found'),
        'success' => false
    ]);

    return;
}

$verificationCodeChecker = new Verifier();

try {
    $verificationCodeChecker->verifyVerificationCode($verification, $_REQUEST['hash']);
    $success = true;
} catch (VerificationAlreadyVerifiedException $exception) {
    $errorReason = VerificationErrorReason::ALREADY_VERIFIED;
} catch (VerificationExpiredException $exception) {
    $errorReason = VerificationErrorReason::EXPIRED;
} catch (InvalidVerificationCodeException $exception) {
    $errorReason = VerificationErrorReason::INVALID_CODE;
} catch (Exception $exception) {
    QUI\System\Log::writeException($exception);

    $Engine->assign([
        'msg' => QUI::getLocale()->get('quiqqer/verification', 'message.types.verifier.error.general'),
        'success' => false
    ]);

    return;
}

try {
    /** @var \QUI\Verification\Interface\LinkVerificationHandlerInterface $verificationHandler */
    $verificationHandler = $repository->getVerificationHandler($verification);
} catch (\Throwable $Exception) {
    if ($Exception instanceof Exception) {
        QUI\System\Log::writeDebugException($Exception);
    }

    $Engine->assign([
        'msg' => QUI::getLocale()->get('quiqqer/verification', 'message.types.verifier.error.general'),
        'success' => false
    ]);

    return;
}

// VERIFICATION SUCCESS
if ($success) {
    // execute onSuccess
    try {
        $verificationHandler->onSuccess($verification);
    } catch (Exception $Exception) {
        QUI\System\Log::addError(
            'Verification onSuccess error: "'
            . $verificationHandler::class . '" (UUID: ' . $verification->uuid . ')'
        );

        QUI\System\Log::writeException($Exception);

        $Engine->assign([
            'msg' => QUI::getLocale()->get('quiqqer/verification', 'message.types.verifier.error.general'),
            'success' => false
        ]);

        return;
    }

    // onSuccess redirect
    try {
        $redirect = $verificationHandler->getOnSuccessRedirectUrl($verification);

        if ($redirect) {
            redirect($redirect);
        }
    } catch (Exception $Exception) {
        QUI\System\Log::addError(
            'Verification getOnSuccessRedirectUrl error: "'
            . $verificationHandler::class . '" (UUID: ' . $verification->uuid . ')'
        );

        QUI\System\Log::writeException($Exception);
    }

    try {
        $msg = $verificationHandler->getSuccessMessage($verification);
    } catch (Exception $Exception) {
        QUI\System\Log::addError(
            'Verification getSuccessMessage error: "'
            . $verificationHandler::class . '" (UUID: ' . $verification->uuid . ')'
        );

        QUI\System\Log::writeException($Exception);
    }

    if (empty($msg)) {
        $msg = QUI::getLocale()->get('quiqqer/verification', 'message.types.verifier.success');
    }
// VERIFICATION ERROR
} else {
    // execute onError
    try {
        $verificationHandler->onError($verification, $errorReason);
    } catch (Exception $Exception) {
        QUI\System\Log::addError(
            'Verification onError error: "'
            . $verificationHandler::class . '" (UUID: ' . $verification->uuid . ')'
        );

        QUI\System\Log::writeException($Exception);

        $Engine->assign([
            'msg' => QUI::getLocale()->get('quiqqer/verification', 'message.types.verifier.error.general'),
            'success' => false
        ]);

        return;
    }

    // onError redirect
    try {
        $redirect = $verificationHandler->getOnErrorRedirectUrl($verification, $errorReason);

        if ($redirect) {
            redirect($redirect);
        }
    } catch (Exception $Exception) {
        QUI\System\Log::addError(
            'Verification getOnErrorRedirectUrl error: "'
            . $verificationHandler::class . '" (UUID: ' . $verification->uuid . ')'
        );

        QUI\System\Log::writeException($Exception);
    }

    // get error message
    try {
        $msg = $verificationHandler->getErrorMessage($verification, $errorReason);
    } catch (Exception $Exception) {
        QUI\System\Log::addError(
            'Verification getErrorMessage error: "'
            . $verificationHandler::class . '" (UUID: ' . $verification->uuid . ')'
        );

        QUI\System\Log::writeException($Exception);
    }

    if (empty($msg)) {
        $msg = QUI::getLocale()->get('quiqqer/verification', 'message.types.verifier.error.general');
    }
}

$Engine->assign([
    'msg' => $msg,
    'success' => $success,
    'Verification' => $verification
]);
